package ru.yandex.passport.phone.ownership.edna;

import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.StringEntity;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.dom.JsonBoolean;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.passport.phone.ownership.PhoneContext;
import ru.yandex.passport.phone.ownership.PhoneOwnershipProxy;
import ru.yandex.passport.phone.ownership.StorageRequestUtils;
import ru.yandex.passport.phone.ownership.parse.PhoneParseResult;
import ru.yandex.passport.phone.ownership.parse.PhoneParseStatus;
import ru.yandex.passport.phone.ownership.parse.PhoneParser;
import ru.yandex.search.prefix.StringPrefix;

public class PhoneManualEdnaUpdateHandler implements ProxyRequestHandler {
    private final PhoneOwnershipProxy proxy;

    public PhoneManualEdnaUpdateHandler(final PhoneOwnershipProxy proxy) {
        this.proxy = proxy;
    }

    @Override
    public void handle(final ProxySession session) throws HttpException, IOException {
        long phone = session.params().getLong("phone");
        boolean onlyNew = session.params().getBoolean("only_new", false);
        String luceneMethod = session.params().getString("lucene_method", "update");
        boolean skipValidation = session.params().getBoolean("skip_invalid", true);
        PhoneParseResult parseResult;
        EdnaCallback callback = new EdnaCallback(session, phone, luceneMethod, System.currentTimeMillis());
        if (!skipValidation) {
            try {
                parseResult = PhoneParser.RUSSIA.parse(session.logger(), String.valueOf(phone));
            } catch (Exception e) {
                callback.failed(e);
                return;
            }

            if (parseResult.status() != PhoneParseStatus.OK) {
                session.response(HttpStatus.SC_UNPROCESSABLE_ENTITY);
                return;
            }
        } else {
            parseResult = new PhoneParseResult(null);
        }

        if (onlyNew) {
            PhoneContext context =
                new PhoneContext(proxy, session, parseResult.phone(), String.valueOf(phone), false);
            StorageRequestUtils.searchRequest(context, proxy, new SearchCallback(session, callback, phone));
            return;
        }

        EdnaRequestUtils.sendImsiRequest(
            proxy,
            session,
            phone,
            callback);
    }

    private class SearchCallback extends AbstractProxySessionCallback<JsonObject> {
        private final EdnaCallback callback;
        private final long phone;

        public SearchCallback(final ProxySession session, final EdnaCallback callback, final long phone) {
            super(session);
            this.callback = callback;
            this.phone = phone;
        }

        @Override
        public void completed(final JsonObject hits) {
            try {
                JsonList hitsArray = hits.asMap().getList("hitsArray");
                if (hitsArray.size() > 0) {
                    session.response(
                        HttpStatus.SC_ACCEPTED,
                        new StringEntity(JsonType.NORMAL.toString(hitsArray.get(0))));
                    return;
                }

                EdnaRequestUtils.sendImsiRequest(
                    proxy,
                    session,
                    phone,
                    callback);
            } catch (Exception e) {
                failed(e);
                return;
            }
        }
    }

    private class EdnaCallback extends AbstractProxySessionCallback<ImsiResponse> {
        private final Long phone;
        private final long ts;
        private final String method;

        public EdnaCallback(final ProxySession session, final Long phone, final String method, final long ts) {
            super(session);
            this.phone = phone;
            this.ts = ts;
            this.method = method;
        }

        public void completed(final ImsiResponse imsiResponse, final Exception e) {
            boolean error = false;
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = JsonType.NORMAL.create(sbw)) {
                writer.startObject();
                writer.key("prefix");
                writer.value(phone);
                writer.key("docs");
                writer.startArray();
                writer.startObject();
                writer.key("id");
                writer.value(PhoneOwnershipProxy.phoneId(phone));
                writer.key("last_edna_request_time");
                writer.value(ts);
                if (e != null ||
                        !imsiResponse.code().equals(AddressRequestStatus.OK) ||
                        !imsiResponse.subscriberCode().equals(AddressRequestStatus.OK))
                {
                    error = true;
                    if (e != null) {
                        writer.key("last_edna_request_error_type");
                        writer.value("http");
                        writer.key("last_edna_request_error_code");
                        writer.value(e.getClass().getName() + ":" + e.getMessage());
                    } else {
                        writer.key("last_edna_request_error_type");
                        writer.value("edna");
                        writer.key("last_edna_request_error_code");
                        if (imsiResponse.subscriberCode() == null) {
                            writer.value(imsiResponse.code());
                        } else {
                            writer.value(imsiResponse.subscriberCode());
                        }
                    }
                } else {
                    writer.key("imsi");
                    writer.value(imsiResponse.imsi());
                    writer.key("on_monitoring");
                    writer.value(JsonBoolean.TRUE);
                    writer.key("last_edna_request_error_type");
                    writer.nullValue();
                    writer.key("last_edna_request_error_code");
                    writer.nullValue();
                }

                writer.endObject();
                writer.endArray();
                writer.endObject();
            } catch (IOException ioe) {
                super.failed(ioe);
                return;
            }

            String data = sbw.toString();
            ProducerClient client = proxy.producerClient().adjust(session.context());
            User user = new User(PhoneOwnershipProxy.PHONE_QUEUE_NAME, new StringPrefix(String.valueOf(phone)));


            client.execute(
                proxy.producerHost(),
                new BasicAsyncRequestProducerGenerator(
                    "/" + method + "?edna_manual&service=" + user.service()
                        + "&prefix=" + user.prefix().toStringFast(),
                    data),
                EmptyAsyncConsumerFactory.ANY_GOOD,
                session.listener().createContextGeneratorFor(client),
                new ProducerCallback(session, data, error));
        }

        @Override
        public void completed(final ImsiResponse imsiResponse) {
            completed(imsiResponse, null);
        }

        @Override
        public void failed(final Exception e) {
            session.logger().log(Level.WARNING, "Edna request failed", e);
            completed(null, e);
        }
    }

    private static final class ProducerCallback extends AbstractProxySessionCallback<Object> {
        private final String document;
        private final boolean error;

        public ProducerCallback(final ProxySession session, final String document, final boolean error) {
            super(session);
            this.document = document;
            this.error = error;
        }

        @Override
        public void completed(final Object o) {
            if (error) {
                session.response(HttpStatus.SC_INTERNAL_SERVER_ERROR, new StringEntity(document, StandardCharsets.UTF_8));
            } else {
                session.response(HttpStatus.SC_OK, new StringEntity(document, StandardCharsets.UTF_8));
            }
        }
    }
}
