package ru.yandex.passport.phone.ownership.parse;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import ru.yandex.logger.PrefixedLogger;

public class MobileOperatorMapping {
    private static final String FILENAME = "DEF-9xx.csv";

    private final List<Range> ranges;
    private final List<MobileOperator> operators;
    private final PrefixedLogger logger;

    public MobileOperatorMapping(final PrefixedLogger logger) throws IOException {
        this.logger = logger;
        List<MobileOperator> operators = new ArrayList<>();
        List<Range> ranges = new ArrayList<>();
        List<String> lines = Files.readAllLines(Path.of(FILENAME));
        for (int i = 1; i < lines.size(); i++) {
            String line = lines.get(i).trim();
            if (line.isEmpty()) {
                continue;
            }

            String[] split = line.split(";");
            String codeStr = split[0];
            String start = split[1];
            String end = split[2];

            //String capacity = split[3];
            String name = split[4];
            String region = split[5];
            long left = Long.parseLong("7" + codeStr + start);
            long right = Long.parseLong("7" + codeStr + end);
            MobileOperator operator = new MobileOperator(left, right, name, region);

            ranges.add(new Range(left, right, operator));
            operators.add(operator);
        }

        this.ranges = Collections.unmodifiableList(ranges);
        this.operators = Collections.unmodifiableList(operators);
    }

    public MobileOperator find(final long phone) {
        return find(phone, logger);
    }

    public MobileOperator find(final long phone, final PrefixedLogger logger) {
        int index = Collections.binarySearch(ranges, new Range(phone, phone, null));

        if (index < 0) {
            index = -index -1;
        }
        if (index > 0) {
            index -= 1;
        }

        Range range = ranges.get(index);
        if (range.left <= phone && range.right >= phone) {
            return range.operator;
        } else {
            if (logger != null) {
                logger.warning("Operator not found " + phone + "most close range is " + range);
            }
        }

        return null;
    }

    private static class Range implements Comparable<Range> {
        private final long left;
        private final long right;
        private final MobileOperator operator;

        public Range(final long left, final long right, final MobileOperator operator) {
            this.left = left;
            this.right = right;
            this.operator = operator;
        }

        @Override
        public int compareTo(final Range o) {
            return Long.compare(left, o.left);
        }

        @Override
        public String toString() {
            return "Range{" +
                       "left=" + left +
                       ", right=" + right +
                       ", operator=" + operator.name() +
                       '}';
        }
    }
}
