package ru.yandex.search.shakur;

import java.io.IOException;
import java.util.ArrayList;
import java.util.LinkedHashSet;
import java.util.Locale;
import java.util.Set;
import java.util.logging.Level;

import org.apache.http.HttpEntity;
import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpHeaders;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;
import org.apache.http.nio.entity.NStringEntity;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;
import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumer;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.server.HttpServer;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumer;
import ru.yandex.json.async.consumer.JsonAsyncTypesafeDomConsumerFactory;
import ru.yandex.json.dom.BasicContainerFactory;
import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonNull;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.StringCollectorsFactory;
import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonTypeExtractor;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.parser.searchmap.User;
import ru.yandex.parser.string.CollectionParser;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.universal.PlainUniversalSearchProxyRequestContext;
import ru.yandex.search.proxy.universal.UniversalSearchProxyRequestContext;

public class CheckPasswordHandler implements HttpAsyncRequestHandler<JsonObject> {
    private static final int MAX_SHA1 = 200;
    private static final boolean ALLOW_LAGGING_HOSTS = true;

    private final Shakur shakur;
    private final String shakurService;
    private final User zeroUser;

    public CheckPasswordHandler(final Shakur shakur)
        throws IOException
    {
        this.shakur = shakur;
        shakurService = shakur.config().shakurService();
        zeroUser =
            new User(shakurService, new LongPrefix(0));
    }

    @Override
    public HttpAsyncRequestConsumer<JsonObject> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException, IOException {
        if (request instanceof HttpEntityEnclosingRequest) {
            HttpEntity entity = ((HttpEntityEnclosingRequest) request).getEntity();
            return new JsonAsyncTypesafeDomConsumer(
                entity,
                StringCollectorsFactory.INSTANCE,
                BasicContainerFactory.INSTANCE);
        } else {
            return new EmptyAsyncConsumer<>(JsonNull.INSTANCE);
        }
    }

    @Override
    public void handle(
        final JsonObject data,
        final HttpAsyncExchange httpAsyncExchange,
        final HttpContext httpContext)
        throws HttpException, IOException
    {
        ProxySession session = new BasicProxySession(shakur, httpAsyncExchange, httpContext);
        CheckPasswordRequestContext context;
        if (data != null && data != JsonNull.INSTANCE) {
            context = new CheckPasswordRequestContext(data, session, shakur.config());
        } else {
            context = new CheckPasswordRequestContext(session, shakur.config());
        }

        AsyncClient client = shakur.searchClient().adjust(session.context());
        UniversalSearchProxyRequestContext requestContext =
            new PlainUniversalSearchProxyRequestContext(
                zeroUser,
                null,
                context.allowLaggingHosts,
                client,
                session.logger());
        shakur.sequentialRequest(
            session,
            requestContext,
            new BasicAsyncRequestProducerGenerator(context.query()),
            context.failoverDelay,
            context.localityShuffle,
            JsonAsyncTypesafeDomConsumerFactory.OK,
            session.listener().createContextGeneratorFor(client),
            new CheckPasswordCallback(session, context, shakur));
    }

    private static class CheckPasswordRequestContext {
        private final boolean localityShuffle;
        private final boolean allowLaggingHosts;
        private final long failoverDelay;
        private final Set<String> sha1;
        private final JsonType jsonType;
        private final int length;
        private final int offset;
        private final boolean passport;
        private final int minFreq;

        CheckPasswordRequestContext(
            final ProxySession session,
            final ImmutableShakurConfig config)
            throws HttpException
        {
            this.localityShuffle = session.params().getBoolean(
                "locality-shuffle",
                true);
            this.failoverDelay = session.params().getLong(
                "failover-delay",
                100L);
            this.allowLaggingHosts = session.params().getBoolean(
                "allow-lagging-hosts",
                ALLOW_LAGGING_HOSTS);
            this.jsonType = JsonTypeExtractor.NORMAL.extract(session.params());
            this.sha1 = session.params().get(
                "sha1",
                new LinkedHashSet<>(),
                    new CollectionParser<>(
                        x -> x.trim().toUpperCase(Locale.ROOT),
                        LinkedHashSet::new,
                        ','));
            this.length = session.params().getInt("length", 500);
            this.offset = session.params().getInt("offset", 0);
            String serviceId =
                (String) session.context().getAttribute(HttpServer.SESSION_USER);
            if (serviceId == null) {
                session.logger().info("TVMServiceID: " + serviceId);
            }
            serviceId =
                (String) session.context().getAttribute(HttpServer.SESSION_USER);
            if (serviceId == null) {
                session.logger().info("TVMSrcID: " + serviceId);
            }
            String ua =
                session.headers().getString(HttpHeaders.USER_AGENT, null);
            if ("python-requests/2.25.1".equals(ua)
                || "python-requests/2.27.1".equals(ua))
            {
                session.logger().info("Passport request");
                this.passport = true;
            } else {
                this.passport = false;
            }
            if (passport) {
                this.minFreq = config.passportMinFreq();
            } else {
                this.minFreq = 0;
            }
        }

        public CheckPasswordRequestContext(
            final JsonObject postData,
            final ProxySession session,
            final ImmutableShakurConfig config)
            throws HttpException
        {
            this(session, config);

            try {
                for (JsonObject jo: postData.asMap().getList("sha1")) {
                    sha1.add(jo.asString().trim().toUpperCase(Locale.ROOT));
                }
            } catch (JsonException je) {
                throw new BadRequestException("Invalid json data in request", je);
            }

            if (sha1.size() > MAX_SHA1) {
                throw new BadRequestException(
                    "Total size of sha1 supplied " + sha1.size()
                        + " expected less than " + MAX_SHA1);
            }
        }

        private String query() throws BadRequestException {
            QueryConstructor query = new QueryConstructor("/search-checkpassword?");
            StringBuilder queryText = new StringBuilder();
            String sep = "";
            for (String sha1: this.sha1) {
                query.append(sep);
                final int sha1Len = sha1.length() / 2;
                switch (sha1Len) {
                    case 0:
                        throw new BadRequestException("Invalid sha1 parameter "
                            + " length: 0");
                    case 1: case 2: case 3:
                        queryText.append("sha1_p4:(");
                        queryText.append(sha1);
                        queryText.append("*)");
                        break;
                    case 4:
                        queryText.append("sha1_p4:(");
                        queryText.append(sha1);
                        queryText.append(")");
                        break;
                    case 5:
                        queryText.append("sha1_p5:(");
                        queryText.append(sha1);
                        queryText.append(")");
                        break;
                    case 6:
                        queryText.append("sha1_p6:(");
                        queryText.append(sha1);
                        queryText.append(")");
                        break;
                    case 7:
                        queryText.append("sha1_p6:(");
                        queryText.append(sha1);
                        queryText.append("*)");
                        break;
                    case 8:
                        queryText.append("sha1_p8:(");
                        queryText.append(sha1);
                        queryText.append(")");
                        break;
                    default:
                        queryText.append("sha1:(");
                        queryText.append(sha1);
                        queryText.append("*)");
                        break;
                }
                sep = " OR ";
            }
            query.append("text", new String(queryText));
            query.append("length", length);
            if (offset > 0) {
                query.append("offset", offset);
            }
            query.append("get", "sha1,source,freq");
            return query.toString();
        }
    }

    private static class CheckPasswordCallback
        extends AbstractProxySessionCallback<JsonObject>
    {
        private final Shakur shakur;
        private final CheckPasswordRequestContext context;

        CheckPasswordCallback(
            final ProxySession session,
            final CheckPasswordRequestContext context,
            final Shakur shakur)
        {
            super(session);
            this.context = context;
            this.shakur = shakur;
        }

        @Override
        public void completed(final JsonObject o) {
            ArrayList<Password> found = new ArrayList<>();
            int foundCnt = 0;
            int skippedCnt = 0;
            int minFreq = Integer.MAX_VALUE;
            try {
                JsonMap m = o.asMap();
                JsonList hits = m.getList("hitsArray");
                for (JsonObject obj: hits) {
                    JsonMap hit = obj.asMap();
                    try {
                        Password pwd = new Password(hit);
                        foundCnt++;
                        if (pwd.freq < minFreq) {
                            minFreq = pwd.freq;
                        }
                        if (context.passport && context.minFreq > pwd.freq) {
                            skippedCnt++;
                            continue;
                        }
                        found.add(pwd);
                    } catch (JsonException e) {
                        session.logger().log(
                            Level.SEVERE,
                            "Hit parse error",
                            e);
                    }
                }
            } catch (JsonException e) {
                session.logger().log(
                    Level.SEVERE,
                    "Json parse error",
                    e);
            }
            session.logger().info("Found: " + foundCnt + " passwords, skipped: "
                + skippedCnt);
            if (context.passport) {
                if (minFreq == Integer.MAX_VALUE) {
                    minFreq = 0;
                }
                shakur.passportFreq(minFreq);
                shakur.passportCheck();
                if (foundCnt > 0) {
                    shakur.passportMatch();
                    if (skippedCnt == foundCnt) {
                        shakur.passportSkip();
                    }
                }
            }
            StringBuilderWriter sbw = new StringBuilderWriter();
            try (JsonWriter writer = context.jsonType.create(sbw)) {
                writer.startObject();
                writer.key("found");
                if (found.size() > 0) {
                    writer.value(true);
                } else {
                    writer.value(false);
                }
                writer.key("passwords");
                writer.startArray();
                for (Password pwd: found) {
                    writer.startObject();
                    writer.key("sha1");
                    writer.value(pwd.sha1);
                    writer.key("source");
                    writer.value(pwd.source);
                    writer.key("freq");
                    writer.value(pwd.freq);
                    writer.endObject();
                }
                writer.endArray();
                writer.endObject();
                HttpEntity entity =
                    new NStringEntity(
                        sbw.toString(),
                        ContentType.APPLICATION_JSON.withCharset(
                            session.acceptedCharset()));
                session.response(HttpStatus.SC_OK, entity);
            } catch (Exception e) {
                session.logger().log(
                    Level.SEVERE,
                    "Response create error",
                    e);
                failed(new ServiceUnavailableException());
            }
        }
    }

    private static final class Password {
        private final String sha1;
        private final String source;
        private final int freq;

        Password(final JsonMap m) throws JsonException {
            sha1 = m.getString("sha1");
            source = m.getString("source");
            freq = m.getInt("freq");
        }
    }

}

