package ru.yandex.peach;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.util.server.AbstractHttpProxyConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.string.PositiveIntegerValidator;
import ru.yandex.parser.string.PositiveLongValidator;

public abstract class AbstractPeachConfigBuilder
    <T extends AbstractPeachConfigBuilder<T>>
    extends AbstractHttpProxyConfigBuilder<T>
    implements PeachConfig
{
    private HttpHostConfigBuilder searchConfig;
    private HttpHostConfigBuilder indexerConfig;
    private String searchQueryParams;
    private String pkField;
    private String pkPrefix;
    private String seqField;
    private String urlField;
    private String queueField;
    private String payloadField;
    private long shardsUpdateInterval;
    private Map<String, PeachQueueConfigBuilder> queuesConfig;
    private Map<Integer, Integer> localPortsMapping;
    private long shardCount;

    protected AbstractPeachConfigBuilder(final PeachConfig config) {
        super(config);
        searchConfig(config.searchConfig());
        indexerConfig(config.indexerConfig());
        searchQueryParams(config.searchQueryParams());
        pkField(config.pkField());
        pkPrefix(config.pkPrefix());
        seqField(config.seqField());
        urlField(config.urlField());
        queueField(config.queueField());
        payloadField(config.payloadField());
        shardsUpdateInterval(config.shardsUpdateInterval());
        queuesConfig = new HashMap<>();
        for (Map.Entry<String, ? extends PeachQueueConfig> entry
                : config.queuesConfig().entrySet())
        {
            queuesConfig.put(
                entry.getKey(),
                new PeachQueueConfigBuilder(entry.getValue()));
        }
        localPortsMapping(config.localPortsMapping());
        shardCount = config.shardCount();
    }

    protected AbstractPeachConfigBuilder(
        final IniConfig config,
        final PeachConfig defaults)
        throws ConfigException
    {
        super(config, defaults);
        searchConfig = new HttpHostConfigBuilder(
            config.section("search"),
            defaults.searchConfig());
        indexerConfig = new HttpHostConfigBuilder(
            config.section("indexer"),
            defaults.indexerConfig());

        IniConfig peach = config.section("peach");
        searchQueryParams = peach.getString(
            "search-query-params",
            defaults.searchQueryParams());
        pkField = peach.get(
            "pk-field",
            defaults.pkField(),
            NonEmptyValidator.INSTANCE);
        pkPrefix = peach.getString("pk-prefix", defaults.pkPrefix());
        seqField = peach.get(
            "seq-field",
            defaults.seqField(),
            NonEmptyValidator.INSTANCE);
        urlField = peach.get(
            "url-field",
            defaults.urlField(),
            NonEmptyValidator.INSTANCE);
        queueField = peach.get(
            "queue-field",
            defaults.queueField(),
            NonEmptyValidator.INSTANCE);
        payloadField = peach.get(
            "payload-field",
            defaults.payloadField(),
            NonEmptyValidator.INSTANCE);
        shardsUpdateInterval = peach.get(
            "shards-update-interval",
            defaults.shardsUpdateInterval(),
            DurationParser.POSITIVE_LONG);
        shardCount = peach.get(
            "shard-count",
            defaults.shardCount(),
            PositiveLongValidator.INSTANCE);

        PeachQueueConfig defaultQueueConfig =
            defaults.queuesConfig().get(null);
        if (defaultQueueConfig == null) {
            defaultQueueConfig = PeachQueueConfigDefaults.INSTANCE;
        }
        PeachQueueConfigBuilder defaultQueueConfigBuilder =
            new PeachQueueConfigBuilder(peach, defaultQueueConfig);
        queuesConfig = new HashMap<>();
        queuesConfig.put(null, defaultQueueConfigBuilder);
        for (Map.Entry<String, IniConfig> entry
                : peach.sections().entrySet())
        {
            String queue = entry.getKey();
            if (!"backend".equals(queue)) {
                PeachQueueConfig defaultLocalQueueConfig =
                    defaults.queuesConfig().get(queue);
                if (defaultLocalQueueConfig == null) {
                    defaultLocalQueueConfig = defaultQueueConfigBuilder;
                }
                queuesConfig.put(
                    queue,
                    new PeachQueueConfigBuilder(
                        entry.getValue(),
                        defaultLocalQueueConfig));
            }
        }

        IniConfig mapping = config.sectionOrNull("local-ports-mapping");
        if (mapping == null) {
            localPortsMapping(defaults.localPortsMapping());
        } else {
            Set<String> keys = mapping.keys();
            localPortsMapping = new HashMap<>(keys.size() << 1);
            for (String key: keys) {
                Integer from;
                try {
                    from = PositiveIntegerValidator.INSTANCE.parse(key);
                } catch (Exception e) {
                    throw new ConfigException(
                        "Failed to parse port mapping for " + key,
                        e);
                }
                localPortsMapping.put(
                    from,
                    mapping.get(key, PositiveIntegerValidator.INSTANCE));
            }
        }
    }

    @Override
    public HttpHostConfigBuilder searchConfig() {
        return searchConfig;
    }

    public T searchConfig(final HttpHostConfig searchConfig) {
        this.searchConfig = new HttpHostConfigBuilder(searchConfig);
        return self();
    }

    @Override
    public HttpHostConfigBuilder indexerConfig() {
        return indexerConfig;
    }

    public T indexerConfig(final HttpHostConfig indexerConfig) {
        this.indexerConfig = new HttpHostConfigBuilder(indexerConfig);
        return self();
    }

    @Override
    public String searchQueryParams() {
        return searchQueryParams;
    }

    public T searchQueryParams(final String searchQueryParams) {
        this.searchQueryParams = searchQueryParams;
        return self();
    }

    @Override
    public String pkField() {
        return pkField;
    }

    public T pkField(final String pkField) {
        this.pkField = pkField;
        return self();
    }

    @Override
    public String pkPrefix() {
        return pkPrefix;
    }

    public T pkPrefix(final String pkPrefix) {
        this.pkPrefix = pkPrefix;
        return self();
    }

    @Override
    public String seqField() {
        return seqField;
    }

    public T seqField(final String seqField) {
        this.seqField = seqField;
        return self();
    }

    @Override
    public String urlField() {
        return urlField;
    }

    public T urlField(final String urlField) {
        this.urlField = urlField;
        return self();
    }

    @Override
    public String queueField() {
        return queueField;
    }

    public T queueField(final String queueField) {
        this.queueField = queueField;
        return self();
    }

    @Override
    public String payloadField() {
        return payloadField;
    }

    public T payloadField(final String payloadField) {
        this.payloadField = payloadField;
        return self();
    }

    @Override
    public long shardsUpdateInterval() {
        return shardsUpdateInterval;
    }

    public T shardsUpdateInterval(final long shardsUpdateInterval) {
        this.shardsUpdateInterval = shardsUpdateInterval;
        return self();
    }

    @Override
    public long shardCount() {
        return shardCount;
    }

    public T shardCount(final long shardCount) {
        this.shardCount = shardCount;
        return self();
    }

    @Override
    public Map<String, PeachQueueConfigBuilder> queuesConfig() {
        return queuesConfig;
    }

    @Override
    public Map<Integer, Integer> localPortsMapping() {
        return localPortsMapping;
    }

    public T localPortsMapping(final Map<Integer, Integer> localPortsMapping) {
        this.localPortsMapping = new HashMap<>(localPortsMapping);
        return self();
    }
}

