package ru.yandex.peach;

import ru.yandex.http.config.HttpTargetConfig;
import ru.yandex.http.config.HttpTargetConfigBuilder;
import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;
import ru.yandex.parser.string.DurationParser;
import ru.yandex.parser.string.LongMemorySizeParser;
import ru.yandex.parser.string.PositiveIntegerValidator;

public abstract class AbstractPeachQueueConfigBuilder
    <T extends AbstractPeachQueueConfigBuilder<T>>
    implements PeachQueueConfig
{
    private HttpTargetConfigBuilder backendConfig;
    private int concurrency;
    private String deadlineParam;
    private long retriesIntervalMax;
    private long retriesIntervalStart;
    private long rateLimitRetriesIntervalStart;
    private long rateLimitRetriesIntervalMax;
    private int batchSize;
    private long batchMemoryLimit;
    private boolean parallel;

    protected AbstractPeachQueueConfigBuilder(final PeachQueueConfig config) {
        backendConfig = new HttpTargetConfigBuilder(config.backendConfig());
        concurrency = config.concurrency();
        deadlineParam = config.deadlineParam();
        retriesIntervalMax = config.retriesIntervalMax();
        retriesIntervalStart = config.retriesIntervalStart();
        rateLimitRetriesIntervalMax = config.rateLimitRetriesIntervalMax();
        rateLimitRetriesIntervalStart = config.rateLimitRetriesIntervalStart();
        batchSize = config.batchSize();
        batchMemoryLimit = config.batchMemoryLimit();
        parallel = config.parallel();
    }

    protected AbstractPeachQueueConfigBuilder(
        final IniConfig config,
        final PeachQueueConfig defaults)
        throws ConfigException
    {
        backendConfig = new HttpTargetConfigBuilder(
            config.section("backend"),
            defaults.backendConfig());
        concurrency = config.get(
            "concurrency",
            defaults.concurrency(),
            PositiveIntegerValidator.INSTANCE);
        deadlineParam =
            config.getString("deadline-param", defaults.deadlineParam());
        retriesIntervalMax = config.get(
            "retries-interval-max",
            defaults.retriesIntervalMax(),
            DurationParser.POSITIVE_LONG);
        retriesIntervalStart = config.get(
            "retries-interval-start",
            defaults.retriesIntervalStart(),
            DurationParser.POSITIVE_LONG);
        rateLimitRetriesIntervalMax = config.get(
            "rate-limit-retries-interval-max",
            defaults.rateLimitRetriesIntervalMax(),
            DurationParser.POSITIVE_LONG);
        rateLimitRetriesIntervalStart = config.get(
            "rate-limit-retries-interval-start",
            defaults.rateLimitRetriesIntervalStart(),
            DurationParser.POSITIVE_LONG);
        batchSize = config.get(
            "batch-size",
            defaults.batchSize(),
            PositiveIntegerValidator.INSTANCE);
        batchMemoryLimit = config.get(
            "batch-memory-limit",
            defaults.batchMemoryLimit(),
            LongMemorySizeParser.INSTANCE);
        parallel = config.getBoolean("parallel", defaults.parallel());
    }

    protected abstract T self();

    @Override
    public HttpTargetConfigBuilder backendConfig() {
        return backendConfig;
    }

    public T backendConfig(final HttpTargetConfig backendConfig) {
        this.backendConfig = new HttpTargetConfigBuilder(backendConfig);
        return self();
    }

    @Override
    public int concurrency() {
        return concurrency;
    }

    public T concurrency(final int concurrency) {
        this.concurrency = concurrency;
        return self();
    }

    @Override
    public String deadlineParam() {
        return deadlineParam;
    }

    public T deadlineParam(final String deadlineParam) {
        this.deadlineParam = deadlineParam;
        return self();
    }

    @Override
    public long retriesIntervalMax() {
        return retriesIntervalMax;
    }

    public T retriesIntervalMax(final long retriesIntervalMax) {
        this.retriesIntervalMax = retriesIntervalMax;
        return self();
    }

    @Override
    public long retriesIntervalStart() {
        return retriesIntervalStart;
    }

    public T retriesIntervalStart(final long retriesIntervalStart) {
        this.retriesIntervalStart = retriesIntervalStart;
        return self();
    }

    @Override
    public long rateLimitRetriesIntervalStart() {
        return rateLimitRetriesIntervalStart;
    }

    public T rateLimitRetriesIntervalStart(
        final long rateLimitRetriesIntervalStart)
    {
        this.rateLimitRetriesIntervalStart = rateLimitRetriesIntervalStart;
        return self();
    }

    @Override
    public long rateLimitRetriesIntervalMax() {
        return rateLimitRetriesIntervalMax;
    }

    public T rateLimitRetriesIntervalMax(
        final long rateLimitRetriesIntervalMax)
    {
        this.rateLimitRetriesIntervalMax = rateLimitRetriesIntervalMax;
        return self();
    }

    @Override
    public int batchSize() {
        return batchSize;
    }

    public T batchSize(final int batchSize) {
        this.batchSize = batchSize;
        return self();
    }

    @Override
    public long batchMemoryLimit() {
        return batchMemoryLimit;
    }

    public T batchMemoryLimit(final int batchMemoryLimit) {
        this.batchMemoryLimit = batchMemoryLimit;
        return self();
    }

    @Override
    public boolean parallel() {
        return parallel;
    }

    public T parallel(final boolean parallel) {
        this.parallel = parallel;
        return self();
    }
}

