package ru.yandex.peach;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import ru.yandex.http.config.ImmutableHttpHostConfig;
import ru.yandex.http.util.server.ImmutableHttpProxyConfig;
import ru.yandex.parser.config.ConfigException;

public class ImmutablePeachConfig
    extends ImmutableHttpProxyConfig
    implements PeachConfig
{
    private final ImmutableHttpHostConfig searchConfig;
    private final ImmutableHttpHostConfig indexerConfig;
    private final String searchQueryParams;
    private final String pkField;
    private final String pkPrefix;
    private final String seqField;
    private final String urlField;
    private final String queueField;
    private final String payloadField;
    private final long shardsUpdateInterval;
    private final Map<String, ImmutablePeachQueueConfig> queuesConfig;
    private final Map<Integer, Integer> localPortsMapping;
    private final long shardCount;

    public ImmutablePeachConfig(final PeachConfig config)
        throws ConfigException
    {
        super(config);
        searchConfig = new ImmutableHttpHostConfig(config.searchConfig());
        indexerConfig = new ImmutableHttpHostConfig(config.indexerConfig());
        searchQueryParams = config.searchQueryParams();
        pkField = config.pkField();
        pkPrefix = config.pkPrefix();
        seqField = config.seqField();
        urlField = config.urlField();
        queueField = config.queueField();
        payloadField = config.payloadField();
        shardsUpdateInterval = config.shardsUpdateInterval();
        shardCount = config.shardCount();
        queuesConfig = new HashMap<>();
        for (Map.Entry<String, ? extends PeachQueueConfig> entry
                : config.queuesConfig().entrySet())
        {
            queuesConfig.put(
                entry.getKey(),
                new ImmutablePeachQueueConfig(entry.getValue()));
        }
        localPortsMapping =
            Collections.unmodifiableMap(config.localPortsMapping());

        if (searchQueryParams == null) {
            throw new ConfigException("search query params is not set");
        }

        if (pkField == null || pkField.isEmpty()) {
            throw new ConfigException("pk field is not set");
        }

        if (seqField == null || seqField.isEmpty()) {
            throw new ConfigException("seq field is not set");
        }

        if (urlField == null || urlField.isEmpty()) {
            throw new ConfigException("url field is not set");
        }

        if (queueField == null || queueField.isEmpty()) {
            throw new ConfigException("queue field is not set");
        }

        if (payloadField != null && payloadField.isEmpty()) {
            throw new ConfigException("payload field is not set");
        }

        if (shardsUpdateInterval <= 0L) {
            throw new ConfigException("shards update interval is not set");
        }

        if (queuesConfig.isEmpty()) {
            throw new ConfigException("no queues configured");
        }
    }

    @Override
    public ImmutableHttpHostConfig searchConfig() {
        return searchConfig;
    }

    @Override
    public ImmutableHttpHostConfig indexerConfig() {
        return indexerConfig;
    }

    @Override
    public String searchQueryParams() {
        return searchQueryParams;
    }

    @Override
    public String pkField() {
        return pkField;
    }

    @Override
    public String pkPrefix() {
        return pkPrefix;
    }

    @Override
    public String seqField() {
        return seqField;
    }

    @Override
    public String urlField() {
        return urlField;
    }

    @Override
    public String queueField() {
        return queueField;
    }

    @Override
    public String payloadField() {
        return payloadField;
    }

    @Override
    public long shardsUpdateInterval() {
        return shardsUpdateInterval;
    }

    @Override
    public long shardCount() {
        return shardCount;
    }

    @Override
    public Map<String, ImmutablePeachQueueConfig> queuesConfig() {
        return queuesConfig;
    }

    @Override
    public Map<Integer, Integer> localPortsMapping() {
        return localPortsMapping;
    }
}

