package ru.yandex.peach;

import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.logging.Level;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicHttpRequest;
import org.apache.http.protocol.HttpContext;
import org.apache.http.protocol.HttpRequestHandler;

import ru.yandex.collection.Pattern;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.function.GenericAutoCloseableHolder;
import ru.yandex.function.Processable;
import ru.yandex.http.server.sync.BaseHttpServer;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.HeadersParser;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.ServiceUnavailableException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.client.ClientBuilder;
import ru.yandex.http.util.client.Timings;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;
import ru.yandex.io.GenericCloseableAdapter;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;
import ru.yandex.json.dom.TypesafeValueContentHandler;
import ru.yandex.parser.string.NonEmptyValidator;
import ru.yandex.parser.uri.ScanningCgiParams;
import ru.yandex.search.prefix.Prefix;
import ru.yandex.search.prefix.PrefixType;
import ru.yandex.stater.Stater;
import ru.yandex.stater.StatsConsumer;
import ru.yandex.util.string.StringUtils;

public class Peach
    extends BaseHttpServer<ImmutablePeachConfig>
    implements HttpRequestHandler
{
    public static final int IOPRIO = 3000;

    private final ConcurrentMap<ShardInfo, Shard> shards =
        new ConcurrentHashMap<>();
    private final ConcurrentHashMap<ShardInfo, ShardInfo> shardInfos =
        new ConcurrentHashMap<>();
    private final Map<String, PeachQueue> queues = new HashMap<>();
    private final CloseableHttpClient searchClient;
    private final CloseableHttpClient indexerClient;
    private final SharedConnectingIOReactor reactor;
    private final GenericAutoCloseableChain<IOException> chain;
    private final Timer timer;
    private final long shardCount;
    private volatile Map<String, ShardsStat> stats = new HashMap<>();

    public Peach(final ImmutablePeachConfig config) throws IOException {
        super(config);
        shardCount = config.shardCount();
        if (config.payloadField() == null) {
            register(new Pattern<>("", true), this, HttpGet.METHOD_NAME);
        } else {
            register(new Pattern<>("", true), this);
        }
        try (GenericAutoCloseableHolder<
                IOException,
                GenericAutoCloseableChain<IOException>> chain =
                    new GenericAutoCloseableHolder<>(
                        new GenericAutoCloseableChain<>()))
        {
            searchClient = ClientBuilder.createClient(
                config.searchConfig(),
                config.dnsConfig());
            chain.get().add(new GenericCloseableAdapter<>(searchClient));
            indexerClient = ClientBuilder.createClient(
                config.indexerConfig(),
                config.dnsConfig());
            chain.get().add(new GenericCloseableAdapter<>(indexerClient));
            reactor = new SharedConnectingIOReactor(
                config,
                config.dnsConfig(),
                new ThreadGroup(
                    getThreadGroup(),
                    config.name() + "-Client"));
            for (String queueName: config.queuesConfig().keySet()) {
                PeachQueue queue =
                    new PeachQueue(queueName, config, thread, reactor);
                chain.get().add(queue);
                queues.put(queueName, queue);
                stats.put(queueName, new ShardsStat());
            }
            this.chain = chain.release();
        }
        registerStater(new PeachStater());
        timer = new Timer(getName() + "-Timer", true);
        timer.schedule(
            new QueueSizeTask(),
            0L,
            config.shardsUpdateInterval());
    }

    @Override
    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = super.status(verbose);
        Map<String, Object> queues = new HashMap<>(this.queues.size() << 1);
        for (Map.Entry<String, PeachQueue> entry: this.queues.entrySet()) {
            queues.put(
                Objects.toString(entry.getKey()),
                entry.getValue().status(verbose));
        }
        status.put("queues", queues);
        return status;
    }

    @Override
    public void start() throws IOException {
        reactor.start();
        for (PeachQueue queue: queues.values()) {
            queue.start();
        }
        super.start();
    }

    @Override
    @SuppressWarnings("try")
    public void close() throws IOException {
        timer.cancel();
        try (GenericAutoCloseableChain<IOException> chain = this.chain) {
            super.close();
        }
        try {
            for (PeachQueue queue: queues.values()) {
                queue.join();
            }
        } catch (InterruptedException e) {
            Thread.currentThread().interrupt();
        }
    }

    public CloseableHttpClient searchClient() {
        return searchClient;
    }

    public CloseableHttpClient indexerClient() {
        return indexerClient;
    }

    private Shard shard(final ShardInfo info)
        throws BadRequestException, StorageFailureException
    {
        Shard shard = shards.get(info);
        if (shard == null) {
            // Intern shard info
            ShardInfo internedInfo = shardInfos.putIfAbsent(info, info);
            if (internedInfo == null) {
                internedInfo = info;
            }
            // Forbid simultaneous shard initialization
            synchronized (internedInfo) {
                // Double-check
                shard = shards.get(internedInfo);
                if (shard == null) {
                    String queueName = internedInfo.queue();
                    PeachQueue queue = queues.get(queueName);
                    if (queue == null) {
                        throw new BadRequestException(
                            "Queue is not configured: " + queueName);
                    }
                    shard = new Shard(queue, this, internedInfo);
                    Shard old = shards.putIfAbsent(internedInfo, shard);
                    if (old != null) {
                        // Impossible situation
                        shard = old;
                    }
                }
            }
        }
        return shard;
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpResponse response,
        final HttpContext context)
        throws HttpException
    {
        String uri = request.getRequestLine().getUri();
        try {
            new URL(uri);
        } catch (MalformedURLException e) {
            throw new BadRequestException("Failed to parse url: " + uri, e);
        }
        Processable<byte[]> payload;
        if (request instanceof HttpEntityEnclosingRequest) {
            if (config.payloadField() == null) {
                throw new ServerException(
                    HttpStatus.SC_METHOD_NOT_ALLOWED,
                    "payload field not set in config, "
                    + "payload aware requests are forbidden");
            }
            try {
                payload = CharsetUtils.toDecodable(
                    ((HttpEntityEnclosingRequest) request).getEntity());
            } catch (IOException e) {
                throw new ServiceUnavailableException(
                    "Failed to consume request payload",
                    e);
            }
        } else {
            payload = null;
        }
        ScanningCgiParams params = new ScanningCgiParams(request);
        Prefix prefix = params.get("prefix", null, PrefixType.LONG);
        if (prefix == null) {
            prefix = params.get("uid", null, PrefixType.LONG);
            if (prefix == null) {
                prefix = params.get("suid", PrefixType.LONG);
            }
        }
        try {
            Shard shard =
                shard(
                    new ShardInfo(
                        prefix.hash() % shardCount,
                        new HeadersParser(request).get(
                            YandexHeaders.X_PEACH_QUEUE,
                            null,
                            NonEmptyValidator.INSTANCE)));
            shard.addTask(uri, payload);
            shard.markNonEmpty();
        } catch (StorageFailureException e) {
            throw new ServiceUnavailableException(e);
        }
    }

    private static class BasicStat {
        private int shards;
        private long size;
    }

    private static class ShardsStat {
        private final BasicStat totalShardsStat = new BasicStat();
        private final BasicStat fatShardsStat = new BasicStat();
    }

    private class PeachStater implements Stater {
        @Override
        public <E extends Exception> void stats(
            final StatsConsumer<? extends E> statsConsumer)
            throws E
        {
            for (Map.Entry<String, ShardsStat> entry: stats.entrySet()) {
                String queue = entry.getKey();
                String prefix;
                if (queue == null) {
                    prefix = "";
                } else {
                    prefix = StringUtils.concat(queue, '-');
                }
                ShardsStat stat = entry.getValue();
                statsConsumer.stat(
                    StringUtils.concat(prefix, "queue-size_ammv"),
                    stat.totalShardsStat.size);
                statsConsumer.stat(
                    StringUtils.concat(prefix, "non-empty-shards_ammv"),
                    stat.totalShardsStat.shards);
                statsConsumer.stat(
                    StringUtils.concat(prefix, "fat-queue-size_ammv"),
                    stat.fatShardsStat.size);
                statsConsumer.stat(
                    StringUtils.concat(prefix, "fat-shards_ammv"),
                    stat.fatShardsStat.shards);
            }
        }
    }

    private class QueueSizeTask extends TimerTask {
        private int shards = 0;
        private long size = 0;

        private void getQueueInfo(final String queue, final ShardsStat stat) {
            Timings timings = new Timings();
            String queueField = config.queueField();
            String urlField = config.urlField();
            String filterText;
            long queueSize = 0;
            if (queue == null) {
                filterText = urlField + ":*+AND+NOT+" + queueField + ":*";
            } else {
                filterText = queueField + ":*%23" + queue;
            }
            HttpRequest request = new BasicHttpRequest(
                HttpGet.METHOD_NAME,
                    "/printkeys-peach?IO_PRIO=" + IOPRIO
                    + config.searchQueryParams()
                    + "&json-type=dollar"
                    + "&field=" + config.urlField()
                    + "&print-freqs&skip-deleted"
                    + "&max-freq=0"
                    + "&text=" + filterText + '&'
                    + "prefixless-field=" + urlField
                    + "&prefixless-field=" + queueField);
            try (CloseableHttpResponse response = searchClient.execute(
                config.searchConfig().host(),
                request,
                timings.createContext()))
            {
                int status = response.getStatusLine().getStatusCode();
                if (status != HttpStatus.SC_OK) {
                    throw new BadResponseException(request, response);
                }
                JsonMap root = TypesafeValueContentHandler.parse(
                    CharsetUtils.content(response.getEntity())).asMap();
                for (Map.Entry<String, JsonObject> entry: root.entrySet()) {
                    String peachUrl = entry.getKey();
                    int sep = peachUrl.indexOf('#');
                    if (sep == -1) {
                        logger().severe("Invalid peach_url token: " + peachUrl);
                        continue;
                    }
                    long shardNo;
                    try {
                        String shardStr = peachUrl.substring(0, sep);
                        shardNo = Long.parseLong(shardStr);
                    } catch (Exception e) {
                        logger().log(
                            Level.SEVERE,
                            "Can't parse shard number from peach_url"
                                + ": " + peachUrl,
                            e);
                        continue;
                    }
                    long shardSize = entry.getValue().asMap().getLong("freq");
                    Shard shard = shard(new ShardInfo(shardNo, queue));
                    shard.markNonEmpty();
                    ++shards;
                    ++stat.totalShardsStat.shards;
                    stat.totalShardsStat.size += shardSize;
                    size += shardSize;
                    queueSize += shardSize;
                    if (shardSize > shard.queueConfig().batchSize()) {
                        ++stat.fatShardsStat.shards;
                        stat.fatShardsStat.size += shardSize;
                    }
                }
            } catch (Throwable e) {
                logger().log(
                    Level.WARNING,
                        "Failed to retrieve shards list for queue: " + queue,
                        e);
                return;
            }
            logger().fine(
                shards + " shards for queue <" + queue
                + "> retrieved from storage in " + timings
                + ", total size: " + queueSize);
        }

        @Override
        public void run() {
            Map<String, ShardsStat> stats = new HashMap<>();
            size = 0;
            shards = 0;
            for (String queue: queues.keySet()) {
                ShardsStat stat = new ShardsStat();
                stats.put(queue, stat);
                getQueueInfo(queue, stat);
            }
            Peach.this.stats = stats;
            logger().fine(shards + " shards total with size: " + size);
        }
    }
}

