package ru.yandex.peach;

import java.io.IOException;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import ru.yandex.concurrent.NamedThreadFactory;
import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.function.GenericAutoCloseableChain;
import ru.yandex.function.GenericAutoCloseableHolder;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.SharedConnectingIOReactor;

public class PeachQueue implements GenericAutoCloseable<IOException> {
    private final Set<Shard> shards = new LinkedHashSet<>();
    private final Lock shardsLock = new ReentrantLock();
    private final Condition shardsCond =
        shardsLock.newCondition();
    private final ImmutablePeachQueueConfig config;
    private final AsyncClient backendClient;
    private final Worker[] workers;
    private final GenericAutoCloseableChain<IOException> chain;

    // CSOFF: ParameterNumber
    public PeachQueue(
        final String name,
        final ImmutablePeachConfig peachConfig,
        final Thread parentThread,
        final SharedConnectingIOReactor reactor)
        throws IOException
    {
        config = peachConfig.queuesConfig().get(name);
        try (GenericAutoCloseableHolder<
                IOException,
                GenericAutoCloseableChain<IOException>> chain =
                    new GenericAutoCloseableHolder<>(
                        new GenericAutoCloseableChain<>()))
        {
            backendClient = new AsyncClient(reactor, config.backendConfig());
            chain.get().add(backendClient);
            NamedThreadFactory threadFactory =
                new NamedThreadFactory(
                    new ThreadGroup(
                        parentThread.getThreadGroup(),
                        parentThread.getName() + '-' + name + "-Worker"));
            workers = new Worker[config.concurrency()];
            for (int i = 0; i < config.concurrency(); ++i) {
                Worker worker = new Worker(this, threadFactory);
                chain.get().add(worker);
                workers[i] = worker;
            }
            this.chain = chain.release();
        }
    }
    // CSON: ParameterNumber

    @Override
    public void close() throws IOException {
        chain.close();
    }

    public Set<Shard> shards() {
        return shards;
    }

    public Lock shardsLock() {
        return shardsLock;
    }

    public Condition shardsCond() {
        return shardsCond;
    }

    public ImmutablePeachQueueConfig config() {
        return config;
    }

    public AsyncClient backendClient() {
        return backendClient;
    }

    public void start() {
        backendClient.start();
        for (Worker worker: workers) {
            worker.start();
        }
    }

    public void join() throws InterruptedException {
        for (Worker worker: workers) {
            worker.join();
        }
    }

    public Map<String, Object> status(final boolean verbose) {
        Map<String, Object> status = new LinkedHashMap<>();
        status.put("client", backendClient.status(verbose));
        return status;
    }

    public void markNonEmpty(final Shard shard) {
        shardsLock.lock();
        try {
            if (shards.add(shard)) {
                shardsCond.signal();
            }
        } finally {
            shardsLock.unlock();
        }
    }
}

