package ru.yandex.peach;

import java.io.IOException;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.Logger;

import org.apache.http.HttpEntity;
import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpStatus;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.protocol.HttpClientContext;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.message.BasicHttpEntityEnclosingRequest;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.function.Processable;
import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.client.Timings;
import ru.yandex.io.StringBuilderWriter;
import ru.yandex.json.parser.ExceptionHandlingContentHandler;
import ru.yandex.json.parser.JsonException;
import ru.yandex.json.parser.JsonParser;
import ru.yandex.json.parser.KeyInterningStringCollectorsFactory;
import ru.yandex.json.writer.DollarJsonWriter;
import ru.yandex.json.writer.JsonWriter;
import ru.yandex.json.xpath.XPathContentHandler;
import ru.yandex.parser.uri.QueryConstructor;
import ru.yandex.search.result.BasicSearchResult;
import ru.yandex.search.result.SearchResultHandler;
import ru.yandex.util.string.HexStrings;

public class Shard {
    private static final String PREFIX = "prefix";
    private static final String DOCS = "docs";
    private static final String SEQ = "seq";
    private static final String QUEUE = "queue";
    private static final String TASK = "Task #";
    private static final String WITH_URL = " with url '";

    private final AtomicBoolean acquired = new AtomicBoolean();
    private final Lock lock = new ReentrantLock();
    private final PeachQueue queue;
    private final long shard;
    private final String queueName;
    private final ImmutablePeachConfig config;
    private final ImmutablePeachQueueConfig queueConfig;
    private final CloseableHttpClient searchClient;
    private final HttpHost searchHost;
    private final CloseableHttpClient indexerClient;
    private final HttpHost indexerHost;
    private final String shardName;
    private final String searchUri;
    private final ContentType contentType;
    private final String pkPrefix;
    private final Logger logger;
    private long seq;

    public Shard(
        final PeachQueue queue,
        final Peach peach,
        final ShardInfo info)
        throws BadRequestException, StorageFailureException
    {
        this.queue = queue;
        shard = info.shard();
        queueName = info.queue();
        config = peach.config();
        queueConfig = queue.config();
        searchClient = peach.searchClient();
        searchHost = config.searchConfig().host();
        indexerClient = peach.indexerClient();
        indexerHost = config.indexerConfig().host();
        QueryConstructor query =
            new QueryConstructor("/search-peach?json-type=dollar");
        query.sb().append(config.searchQueryParams());
        query.append("IO_PRIO", Peach.IOPRIO);
        long batchMemoryLimit = queueConfig.batchMemoryLimit();
        if (batchMemoryLimit > 0L) {
            query.append("memory-limit", batchMemoryLimit);
        }
        query.append(PREFIX, shard);
        String text;
        if (queueName == null) {
            shardName = Long.toString(shard);
            text =
                config.urlField() + ":1 AND NOT "
                + config.queueField() + ':' + '*';
        } else {
            shardName = Long.toString(shard) + '@' + queueName;
            text =
                config.urlField() + ":1 AND "
                + config.queueField() + ':' + '"' + queueName + '"';
        }
        query.append("text", text);
        query.append("sort", config.seqField());
        String payloadField = config.payloadField();
        String get;
        if (payloadField == null) {
            get =
                config.pkField() + ',' + config.seqField()
                + ',' + config.urlField();
        } else {
            get =
                config.pkField() + ',' + config.seqField()
                + ',' + config.urlField() + ',' + payloadField;
        }
        query.append("get", get);
        String initStringUri = query + "&length=1";
        query.append("length", queueConfig.batchSize());
        searchUri = query + "&asc";
        contentType = ContentType.APPLICATION_JSON.withCharset(
            config.indexerConfig().requestCharset());
        StringBuilder sb = new StringBuilder(config.pkPrefix());
        sb.append(shard);
        if (queueName != null) {
            sb.append('@');
            sb.append(queueName);
        }
        sb.append('_');
        pkPrefix = new String(sb);
        logger = peach.logger().addPrefix("Shard-" + shardName);
        Task[] tasks = getTasks(initStringUri);
        if (tasks.length == 0) {
            seq = 0L;
        } else {
            seq = tasks[0].seq() + 1L;
        }
        logger.info("Seq initialized to " + seq);
    }

    public ImmutablePeachConfig config() {
        return config;
    }

    public ImmutablePeachQueueConfig queueConfig() {
        return queueConfig;
    }

    public void markNonEmpty() {
        queue.markNonEmpty(this);
    }

    public boolean acquire() {
        return acquired.compareAndSet(false, true);
    }

    public void release() {
        acquired.set(false);
    }

    public Logger logger() {
        return logger;
    }

    private Task[] getTasks(final String uri)
        throws StorageFailureException
    {
        Timings timings = new Timings();
        Task[] tasks;
        try (CloseableHttpResponse response = searchClient.execute(
                searchHost,
                new BasicHttpRequest(HttpGet.METHOD_NAME, uri),
                timings.createContext()))
        {
            if (response.getStatusLine().getStatusCode() != HttpStatus.SC_OK) {
                throw new BadResponseException(uri, response);
            }
            HttpEntity entity = response.getEntity();
            BasicSearchResult result = new BasicSearchResult();
            XPathContentHandler handler = new XPathContentHandler(
                new SearchResultHandler(result),
                KeyInterningStringCollectorsFactory.INSTANCE.apply(
                    entity.getContentLength()));
            new JsonParser(
                new ExceptionHandlingContentHandler(handler, handler))
                .parse(CharsetUtils.content(entity));
            tasks = Task.create(result.hitsArray(), config);
        } catch (HttpException | IOException | JsonException e) {
            throw new StorageFailureException(
                "Failed to get data for shard " + shardName,
                e);
        }
        if (tasks.length == 0) {
            logger.info("No tasks found for shard");
        } else {
            logger.info(
                tasks.length + " tasks with #"
                + tasks[0].seq() + '-' + tasks[tasks.length - 1].seq()
                + " retrieved in " + timings);
        }
        return tasks;
    }

    public Task[] getTasks() throws StorageFailureException {
        return getTasks(searchUri);
    }

    public void deleteTask(final Task task) throws StorageFailureException {
        long seq = task.seq();
        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = new DollarJsonWriter(sbw)) {
            writer.startObject();
            writer.key(PREFIX);
            writer.value(shard);
            writer.key(DOCS);
            writer.startArray();
            writer.startObject();
            writer.key(config.pkField());
            writer.value(task.pk());
            writer.key(config.seqField());
            writer.value(seq + 1L);
            writer.endObject();
            writer.endArray();
            writer.endObject();
        } catch (IOException e) {
            throw new StorageFailureException(
                "Failed to construct delete request");
        }
        Timings timings = new Timings();
        try {
            QueryConstructor query =
                new QueryConstructor("/delete?prefix=" + shard);
            query.append(SEQ, seq);
            if (queueName != null) {
                query.append(QUEUE, queueName);
            }
            BasicHttpEntityEnclosingRequest post =
                new BasicHttpEntityEnclosingRequest(
                    HttpPost.METHOD_NAME,
                    query.toString());
            post.setEntity(new StringEntity(sbw.toString(), contentType));
            try (CloseableHttpResponse response = indexerClient.execute(
                    indexerHost,
                    post,
                    timings.createContext()))
            {
                int status = response.getStatusLine().getStatusCode();
                if (status != HttpStatus.SC_OK) {
                    throw new BadResponseException(query, response);
                }
            }
        } catch (HttpException | IOException e) {
            throw new StorageFailureException(
                "Failed to delete task " + seq + " from shard " + shardName,
                e);
        }
        logger.info(TASK + seq + " deleted in " + timings);
    }

    public void addTask(
        final String url,
        final Processable<byte[]> payload)
        throws BadRequestException, StorageFailureException
    {
        Timings timings = new Timings();
        HttpClientContext context = timings.createContext();
        StringBuilder pk = new StringBuilder(pkPrefix);
        QueryConstructor query = new QueryConstructor("/add?prefix=" + shard);
        if (queueName != null) {
            query.append(QUEUE, queueName);
        }
        StringBuilderWriter sbw = new StringBuilderWriter();
        JsonWriter writer = new DollarJsonWriter(sbw);
        long seq;
        try {
            writer.startObject();
            writer.key(PREFIX);
            writer.value(shard);
            writer.key(DOCS);
            writer.startArray();
            writer.startObject();
            writer.key(config.urlField());
            writer.value(url);
            if (queueName != null) {
                writer.key(config.queueField());
                writer.value(queueName);
            }
            if (payload != null) {
                writer.key(config.payloadField());
                writer.startString();
                writer.write(payload.processWith(HexStrings.UPPER));
                writer.endString();
            }
            writer.key(config.pkField());
            lock.lock();
            try {
                seq = this.seq++;
                pk.append(seq);
                writer.value(pk);
                writer.key(config.seqField());
                writer.value(seq);
                writer.endObject();
                writer.endArray();
                writer.endObject();
                writer.close();
                query.append(SEQ, seq);
                BasicHttpEntityEnclosingRequest post =
                    new BasicHttpEntityEnclosingRequest(
                        HttpPost.METHOD_NAME,
                        query.toString());
                post.setEntity(new StringEntity(sbw.toString(), contentType));
                try (CloseableHttpResponse response =
                        indexerClient.execute(indexerHost, post, context))
                {
                    int status = response.getStatusLine().getStatusCode();
                    if (status != HttpStatus.SC_OK) {
                        throw new BadResponseException(query, response);
                    }
                } catch (HttpException | IOException e) {
                    throw new StorageFailureException(
                        "Failed to add url '" + url
                        + "' to shard " + shardName
                        + " with seq " + seq,
                        e);
                }
            } finally {
                lock.unlock();
            }
        } catch (IOException e) {
            throw new BadRequestException(
                "Failed to encode request to JSON",
                e);
        }
        logger.info(TASK + seq + WITH_URL + url + "' added in " + timings);
    }
}

