package ru.yandex.peach;

import java.io.IOException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.logging.Level;

import org.apache.http.entity.ContentType;

import ru.yandex.function.GenericAutoCloseable;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.ServerErrorStatusPredicate;
import ru.yandex.http.util.ServerException;
import ru.yandex.http.util.YandexHttpStatus;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;
import ru.yandex.http.util.nio.client.AsyncGetURIRequestProducerSupplier;
import ru.yandex.http.util.nio.client.AsyncPostURIRequestProducerSupplier;
import ru.yandex.parser.string.URIParser;
import ru.yandex.util.string.UnhexStrings;

public class Worker
    implements GenericAutoCloseable<IOException>, Runnable
{
    private static final long WAIT_INTERVAL = 1000000000L;
    private static final String TASK = "Task #";
    private static final StatusCheckAsyncResponseConsumerFactory<Void>
        CONSUMER_FACTORY =
            new StatusCheckAsyncResponseConsumerFactory<>(
                x -> x != YandexHttpStatus.SC_UNAUTHORIZED
                    && x != YandexHttpStatus.SC_FORBIDDEN
                    && x != YandexHttpStatus.SC_TOO_MANY_REQUESTS
                    && x != YandexHttpStatus.SC_BUSY
                    && !ServerErrorStatusPredicate.INSTANCE.test(x),
                EmptyAsyncConsumerFactory.INSTANCE);

    private final StringBuilder sb = new StringBuilder();
    private final Set<Shard> shards;
    private final Lock shardsLock;
    private final Condition shardsCond;
    private final AsyncClient client;
    private final Thread thread;
    private volatile boolean stopped = false;

    public Worker(final PeachQueue queue, final ThreadFactory threadFactory) {
        shards = queue.shards();
        shardsLock = queue.shardsLock();
        shardsCond = queue.shardsCond();
        client = queue.backendClient();
        thread = threadFactory.newThread(this);
    }

    public void start() {
        thread.start();
    }

    public void join() throws InterruptedException {
        thread.join();
    }

    @Override
    public void close() {
        stopped = true;
        thread.interrupt();
    }

    private static boolean isLocalhost(final String host) {
        try {
            InetAddress addr = InetAddress.getByName(host);
            return addr.isAnyLocalAddress()
                || addr.isLoopbackAddress()
                || NetworkInterface.getByInetAddress(addr) != null;
        } catch (SocketException | UnknownHostException e) {
            return false;
        }
    }

    private URI uri(
        final Shard shard,
        final Task task,
        final String suffix)
        throws StorageFailureException
    {
        String url = task.url();
        try {
            URI uri = new URI(url).parseServerAuthority();
            int port = uri.getPort();
            Integer mappedPort = shard.config().localPortsMapping().get(port);
            if (mappedPort != null
                && mappedPort.intValue() != port
                && isLocalhost(uri.getHost()))
            {
                shard.logger().info(
                    "Local port " + port + " mapped to " + mappedPort);
                port = mappedPort;
            }
            sb.setLength(0);
            sb.append(uri.getRawQuery());
            sb.append("&task-seq=");
            sb.append(task.seq());
            sb.append(suffix);
            return URIParser.createURI(
                uri.getScheme(),
                uri.getRawUserInfo(),
                uri.getHost(),
                port,
                uri.getRawPath(),
                new String(sb),
                null,
                uri.getRawSchemeSpecificPart(),
                uri.getRawAuthority());
        } catch (URISyntaxException e) {
            throw new StorageFailureException(
                TASK + task.seq() + " has bad url '" + url + '\'',
                e);
        }
    }

    private Future<Void> spawnRequest(
        final Shard shard,
        final Task task,
        final String suffix)
        throws StorageFailureException
    {
        URI uri = uri(shard, task, suffix);
        String payload = task.payload();
        if (payload == null) {
            return client.execute(
                new AsyncGetURIRequestProducerSupplier(uri),
                CONSUMER_FACTORY,
                EmptyFutureCallback.INSTANCE);
        } else {
            return client.execute(
                new AsyncPostURIRequestProducerSupplier(
                    uri,
                    UnhexStrings.unhex(payload),
                    ContentType.APPLICATION_JSON),
                CONSUMER_FACTORY,
                EmptyFutureCallback.INSTANCE);
        }
    }

    private static boolean checkRateLimit(final ExecutionException e) {
        Throwable cause = e.getCause();
        if (cause instanceof ServerException) {
            int code = ((ServerException) cause).statusCode();
            if (code == YandexHttpStatus.SC_TOO_MANY_REQUESTS) {
                return true;
            }
        }
        return false;
    }

    private void processTasks(final Shard shard, final Task[] tasks)
        throws StorageFailureException
    {
        long now = System.currentTimeMillis();
        ImmutablePeachQueueConfig queueConfig = shard.queueConfig();
        String suffix;
        String deadlineParam = queueConfig.deadlineParam();
        if (deadlineParam.isEmpty()) {
            suffix = "";
        } else {
            long deadline = now + queueConfig.backendConfig().timeout();
            // Url will always contain query, because prefix is the
            // required param
            suffix = '&' + deadlineParam + '=' + deadline;
        }
        List<Future<Void>> futures = new ArrayList<>(tasks.length);
        for (Task task: tasks) {
            futures.add(spawnRequest(shard, task, suffix));
        }
        for (int i = 0; i < tasks.length; ++i) {
            int retries = 0;
            Task task = tasks[i];
            long rateLimitRetryDelay =
                queueConfig.rateLimitRetriesIntervalStart();
            long retryDelay =
                queueConfig.retriesIntervalStart();
            Future<Void> future = futures.get(i);
            do {
                try {
                    try {
                        future.get();
                        future = null;
                    } catch (ExecutionException e) {
                        shard.logger().log(
                            Level.WARNING,
                            "Failed to process task #" + task.seq()
                            + '(' + i + ')' + " with request: " + task.url()
                            + ", retries failed so far: " + retries,
                            e);

                        ++retries;
                        if (checkRateLimit(e)) {
                            shard.logger().info(
                                "Hit rate limit, delaying for "
                                + rateLimitRetryDelay + " ms");
                            Thread.sleep(rateLimitRetryDelay);
                            rateLimitRetryDelay =
                                Math.min(
                                    rateLimitRetryDelay << 1,
                                    queueConfig.rateLimitRetriesIntervalMax());
                        } else {
                            retryDelay =
                                Math.min(
                                    retryDelay << 1,
                                    queueConfig.retriesIntervalMax());
                            Thread.sleep(retryDelay);
                        }
                        String retrySuffix;
                        if (deadlineParam.isEmpty()) {
                            retrySuffix = "";
                        } else {
                            long deadline =
                                System.currentTimeMillis()
                                + queueConfig.backendConfig().timeout();
                            retrySuffix = '&' + deadlineParam + '=' + deadline;
                        }

                        future = spawnRequest(shard, task, retrySuffix);
                        shard.logger().info(TASK + task.seq() + " respawned");
                    }
                } catch (InterruptedException e) {
                    shard.logger().log(
                        Level.WARNING,
                        "Tasks processing interrupted",
                        e);
                    return;
                }
            } while (future != null);
            shard.deleteTask(task);
        }
        shard.logger().info(
            tasks.length + " tasks with #"
            + tasks[0].seq() + '-' + tasks[tasks.length - 1].seq()
            + " processed in " + (System.currentTimeMillis() - now)
            + " ms");
    }

    private boolean processShard(final Shard shard)
        throws StorageFailureException
    {
        Task[] tasks = shard.getTasks();
        if (tasks.length == 0) {
            return false;
        } else {
            if (shard.queueConfig().parallel()) {
                processTasks(shard, tasks);
            } else {
                for (Task task: tasks) {
                    processTasks(shard, new Task[]{task});
                }
            }
            return true;
        }
    }

    @Override
    public void run() {
        while (!stopped) {
            Shard shard = null;
            shardsLock.lock();
            try {
                Iterator<Shard> iter = shards.iterator();
                if (iter.hasNext()) {
                    shard = iter.next();
                    iter.remove();
                }
                if (shard == null) {
                    try {
                        shardsCond.awaitNanos(WAIT_INTERVAL);
                    } catch (InterruptedException e) {
                    }
                    continue;
                }
            } finally {
                shardsLock.unlock();
            }
            if (shard.acquire()) {
                boolean returnToShards = true;
                try {
                    returnToShards = processShard(shard);
                } catch (Exception e) {
                    // IllegalArgumentException could be thrown
                    // from HexStrings.unhex
                    // just in case - not exiting the thread
                    shard.logger().log(
                        Level.WARNING,
                        "Shard processing failed",
                        e);
                } finally {
                    shard.release();
                    if (returnToShards) {
                        shardsLock.lock();
                        try {
                            shards.add(shard);
                        } finally {
                            shardsLock.unlock();
                        }
                        shard.logger().info(
                            "Shard returned to non-empty shards list");
                    }
                }
            }
        }
    }
}

