import asyncpg
from aiohttp import web

from ..misc.exceptions import BackpackAPIServerHardError


class BackupInit:

    def __init__(self, app):
        self.app = app
        self.db = self.app["db"]
        self.log = self.app["alog"]

    async def backupInit(self, request):
        self.log.info(f'{__name__} called')

        rq = await request.json()
        meta = rq['meta']
        meta_shard = meta['shard']
        meta_inum = meta['inum']
        meta_hostname = meta['hostname']
        infotable = "{}_{}".format(meta['service'], meta['version'])
        services_meta_table = "services_meta"
        services_table = "services"
        fileinfo = rq['fileinfo']

        s_id = await self.db.fetchval(
            '''SELECT s_id FROM {s_table} WHERE service = $1'''.format(s_table=services_table),
            meta['service'])

        self.log.info(f"Init service id is {s_id}")

        if s_id is None:
            raise BackpackAPIServerHardError("Service not found in meta table. Add service firstly!")

        # Select b_id on exception - create and return
        try:
            b_id = await self.db.fetchval(
                '''SELECT b_id FROM {services_meta_table} WHERE version = $1 AND service_id = $2'''
                .format(services_meta_table=services_meta_table),
                int(meta['version']),
                s_id)
            if not b_id:
                self.log.info(f"Row for backup meta files {services_meta_table} doesnot exist, creating.".format(
                    services_meta_table))
                b_id = await self.db.fetchval(
                    '''INSERT INTO {services_meta_table}(service_id, version) VALUES($1, $2) RETURNING b_id'''
                        .format(services_meta_table=services_meta_table),
                    s_id,
                    (int(meta['version']))
                )
        except asyncpg.exceptions.UndefinedTableError:
            raise BackpackAPIServerHardError(f"Metatable {services_meta_table} does not exists, please create firstly")

        self.log.info("Info table for backup: {}".format(infotable))
        self.log.info("Backup id is: {}".format(b_id))

        # TODO: On every backup we delete all information about shard. But if backup started for some another hostname -
        #  new files will be unserted for this shard. We can drop all files for every hostname. But there we can lost some
        #  metadata.

        try:

            # Set stalled = True for all records for this shard and inum
            stalled = await self.db.execute(
                '''UPDATE {infotable} SET stalled = $1 WHERE shard = $2 AND inum = $3'''.format(
                    infotable=infotable),
                True,
                (int(meta_shard)),
                (int(meta_inum))
            )
            self.log.info(
                f"Set stalled for shard, all old files for all hosts: {meta_shard} table: {infotable}. Stalled: {stalled}")

            # Drop all records from prevous backup.
            # Maybe we doesnot need this because old files for shard will be marked as Stalled
            if self.app['dropOnInit'] == "True":
                deleted = await self.db.execute(
                    '''DELETE FROM {infotable} WHERE shard = $1 AND inum = $2 AND hostname = $3'''.format(
                        infotable=infotable),
                    (int(meta_shard)),
                    (int(meta_inum)),
                    (meta_hostname))
                self.log.info(
                    f"Drop on init config specified. Delete all infos on init foe shard: {meta_shard} table: {infotable}. Deleted: {deleted}")
        except asyncpg.exceptions.UndefinedTableError:
            self.log.info(f"Table: {infotable} doesnot exist, drop on init operation passed")

        for path, info in fileinfo.items():
            # TODO: We need drop all infos about this shard if there some information,
            #  because new files may be different
            self.log.info("path is: {}".format(path))

            try:
                f_id = await self.initfileinfo(infotable, path, info, b_id)
            except asyncpg.exceptions.UndefinedTableError:

                # Uniq keys for init table

                self.log.info("Table  for files {} doesnot exist, creating.".format(infotable))

                await self.db.execute('''
                    CREATE TABLE {infotable}(
                    f_id bigserial not null primary key,
                    path varchar(5000) not null,
                    rootpath varchar(5000) not null,
                    msg varchar(2000) not null,
                    mdskey varchar(5000) not null,
                    status varchar(200) not null,
                    md5 varchar(200),
                    fsize bigint not null,
                    shard integer not null,
                    numdocs integer not null,
                    inum integer not null,
                    b_id integer not null,
                    hostname varchar(1000) not null,
                    stalled boolean default false not null,
                    UNIQUE (path, md5, hostname)
                    )
                '''.format(infotable=infotable))

                f_id = await self.initfileinfo(infotable, path, info, b_id)

            self.log.info("File backup id is: {}".format(f_id))

        return web.Response(text="OK", status=200)

    async def initfileinfo(self, infotable, path, info, b_id):
        try:
            f_id = await self.db.fetchval(
                '''INSERT INTO {infotable}(path, rootpath, msg, mdskey, status, md5, fsize, shard, numdocs, inum, hostname, b_id)
                VALUES($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12) ON CONFLICT (path, md5, hostname) DO UPDATE SET stalled = $13 RETURNING f_id'''.format(infotable=infotable),
                (path),
                (info['rootpath']),
                (info['msg']),
                (info['mdskey']),
                (info['status']),
                (info['md5']),
                (int(info['fsize'])),
                (int(info['shard'])),
                (int(info['numdocs'])),
                (int(info['inum'])),
                (info['hostname']),
                (int(b_id)),
                False
            )
        except asyncpg.exceptions.UniqueViolationError:
            self.log.warning("Table init pass. This records was already initialized: {} {} {}".format(path,
                                                                                                      info['md5'],
                                                                                                      info['hostname']
                                                                                                      ))
            f_id = "none"
        return f_id
