import asyncio

import aiohttp_jinja2
import asyncpg
import ssl
import socket

import logging.handlers
import os
from yaml import load, SafeLoader
from optparse import OptionParser

import jinja2
from aiohttp import web
from aiohttp.web_app import Application

from .handlers.SlotUpdate import SlotUpdate
from .handlers.SlotRequest import SlotRequest
from .handlers.SlotAdd import SlotAdd
from .handlers.ServiceBackupInfo import ServiceBackupInfo
from .handlers.ShardBackupInfo import ShardBackupInfo
from .backend.PostgresDb import PostgresDb
from .handlers.Ping import Ping

from .handlers.views.ServiceBackupInumView import ServiceBackupInumView
from .handlers.views.ServiceBackupShardDetailView import ServiceBackupShardDetailView
from .handlers.views.ServiceBackupShardView import ServiceBackupShardView
from .handlers.views.ServiceDetailBackupVersionView import ServiceDetailBackupVersionView
from .handlers.views.ServiceBackupVersionView import ServiceBackupVersionView
from .handlers.views.ServiceView import ServiceView
from .handlers.views.WorkerView import WorkerView
from .handlers.views.ServiceDetailView import ServiceDetailView

from .handlers.FileInfo import FileInfo
from .handlers.FileRestoreInfoUpdate import FileRestoreInfoUpdate
from .handlers.FileInfoUpdate import FileInfoUpdate
from .handlers.BackupInit import BackupInit

import uvloop
asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())


def load_config(config):
    with open(config) as f:
        conf = load("".join(f.readlines()),  Loader=SafeLoader)
    return conf


def full_withEnv(stringvar):
    stringvar = stringvar % os.environ
    return stringvar


def main():
    parser = OptionParser()
    parser.add_option("-c", "--config", type="string", dest="config",
                      metavar="CONFIG", default="config_qa.yaml",
                      help="YAML config file (default: %default)")
    (options, args) = parser.parse_args()

    conf = load_config(options.config)

    connectUri = full_withEnv(conf['database']['connectUri'])
    # name = full_withEnv(conf['base']['name'])

    httpServerPort = int(full_withEnv(conf['server']['httpServerPort']))
    # serverThreads = int(full_withEnv(conf['server']['serverThreads']))
    # oauth_token = str(full_withEnv(conf['settings']['oauth_token']))
    full_log_path = full_withEnv(conf['log']['fullLogPath'])
    access_log_path = full_withEnv(conf['log']['accessLogPath'])
    debug_log_path = full_withEnv(conf['log']['debugLogPath'])
    log_format = conf['log']['format']
    stdout_log = conf['log']['stdout']


#   templatesDir = os.path.join(os.path.dirname(__file__), conf['settings']['templatesDir'])
    templatesDir = conf['settings']['templatesDir']
#   staticDir = os.path.join(os.path.dirname(__file__), conf['settings']['staticDir'])
    staticDir = conf['settings']['staticDir']

    caCertFile = full_withEnv(conf['settings']['caCert'])

    debug = full_withEnv(conf['settings']['debug'])
    dropOnInit = full_withEnv(conf['special']['dropOnInit'])

    if stdout_log == "False":
        stdout_log = False

    logging.basicConfig(level=logging.DEBUG,
                        format=log_format)

    full_filehandler = logging.handlers.WatchedFileHandler(full_log_path)
    full_filehandler.setFormatter(logging.Formatter(log_format))

    access_filehandler = logging.handlers.WatchedFileHandler(access_log_path)
    access_filehandler.setFormatter(logging.Formatter(log_format))

    debug_filehandler = logging.handlers.WatchedFileHandler(debug_log_path)
    debug_filehandler.setFormatter(logging.Formatter(log_format))

    access_log = logging.getLogger("aiohttp.access")
    server_log = logging.getLogger("aiohttp.server")

    debug_log = logging.getLogger('asyncio')

    access_log.propagate = stdout_log
    server_log.propagate = stdout_log
    debug_log.propagate = stdout_log

    access_log.addHandler(access_filehandler)
    server_log.addHandler(full_filehandler)
    debug_log.addHandler(debug_filehandler)

    async def init_app():
        """Initialize the application server."""

        ctx = ssl.create_default_context(cafile=caCertFile)
        ctx.check_hostname = False
        ctx.verify_mode = ssl.CERT_NONE

        app: Application = web.Application()

        aiohttp_jinja2.setup(app,
                             loader=jinja2.FileSystemLoader(templatesDir))

        pool = await asyncpg.create_pool(
            dsn=connectUri,
            ssl=ctx)

        app['db'] = PostgresDb(pool)
        app['alog'] = access_log
        app['slog'] = server_log

        app['dropOnInit'] = dropOnInit

        app.router.add_route('POST', '/v1/backup/init', BackupInit(app).backupInit)
        app.router.add_route('POST', '/v1/backup/file/update', FileInfoUpdate(app).fileInfoUpdate)

        app.router.add_route('POST', '/v1/restore/file/update', FileRestoreInfoUpdate(app).fileRestoreInfoUpdate)

        app.router.add_route('GET', '/v1/backup/info', FileInfo(app).fileInfo)
        app.router.add_route('GET', '/v1/slot/request', SlotRequest(app).request)
        app.router.add_route('GET', '/v1/slot/update', SlotUpdate(app).request)
        app.router.add_route('GET', '/v1/slot/add', SlotAdd(app).request)
        app.router.add_route('GET', '/v1/backup/shardinfo', ShardBackupInfo(app).backupInfo)
        app.router.add_route('GET', '/v1/backup/serviceinfo', ServiceBackupInfo(app).backupInfo)
        app.router.add_route('GET', '/ping', Ping(app).ping)

        app.router.add_static('/static', staticDir)

        app.router.add_view('/', ServiceView)
        app.router.add_view('/detail', ServiceDetailView)
        app.router.add_view('/service', ServiceView)
        app.router.add_view('/worker', WorkerView)
        app.router.add_view('/service/detail', ServiceDetailView)
        app.router.add_view('/backup/{service_name}', ServiceBackupVersionView)
        app.router.add_view('/backup/{service_name}/detail', ServiceDetailBackupVersionView)
        app.router.add_view('/backup/{service_name}/{version}', ServiceBackupInumView)
        app.router.add_view('/backup/{service_name}/{version}/{inum}', ServiceBackupShardView)
        app.router.add_view('/backup/{service_name}/{version}/{inum}/{shard}', ServiceBackupShardDetailView)

        return app

    # Listen on both IPv4 and IPv6
    sock = socket.socket(socket.AF_INET6)
    sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEPORT, 1)
    sock.bind(('::', httpServerPort))

    loop = asyncio.get_event_loop()
    # loop = uvloop.new_event_loop()
    # loop.set

    access_log.info(f'debug {debug}')
    #
    if debug == "True":
        loop.set_debug(True)

    inited_app = loop.run_until_complete(init_app())
    web.run_app(inited_app, sock=sock, reuse_address=True, reuse_port=True)


if __name__ == '__main__':
    main()
