import asyncio
import time
import aiohttp
import logging.handlers
from concurrent.futures import ProcessPoolExecutor
import os
import socket
import async_timeout
# import sys
from random import randint
from yaml import load, SafeLoader
from optparse import OptionParser
from .misc.libs import get_searchmap_parsed
from .misc.exceptions import BackpackWorkerStatusError
from .misc.defaults import STATUS_WORKER_FINISHED, STATUS_WORKER_TRY_LATER, STATUS_ERROR, STATUS_UNKNOWN

INFO_URL = "http://backpack-meta.mail.yandex.net/v1/backup/shardinfo"
LOCK_URL = "http://backpack-meta.mail.yandex.net/v1/slot/request"
UPDATE_LOCK_URL = "http://backpack-meta.mail.yandex.net/v1/slot/update"


class Worker:

    def __init__(self, defaultconfig):

        conf = self.load_config(defaultconfig)

        full_log_path = self.full_withEnv(conf['log']['fullLogPath'])
        log_format = conf['log']['format']
        stdout_log = conf['log']['stdout']
        logging.basicConfig(level=logging.INFO,
                            format=log_format)
        full_filehandler = logging.handlers.WatchedFileHandler(full_log_path)
        full_filehandler.setFormatter(logging.Formatter(log_format))
        self.log = logging.getLogger('Worker')
        if stdout_log == "False":
            stdout_log = False
        self.log.propagate = stdout_log
        self.log.addHandler(full_filehandler)

        # self.version = int(time.time())
        # Check lock and get config
        # TODO: get some uniq key from env unstead of hostname?
        self.hostname = socket.gethostname()
        # B_LOCK_URL = f'{LOCK_URL}?hostname={self.hostname}&timestamp={int(time.time())}&version={self.version}'

        lock_loop = asyncio.new_event_loop()
        # flag, self.version, data = lock_loop.run_until_complete(self.getlock(f_timeout=20, t_timeout=120))
        self.version, data = lock_loop.run_until_complete(self.getlock(f_timeout=20, t_timeout=120))

        # if not flag:
        #     self.log.info(f"Cannot obtain lock - retries number exceded, Exit - need to get stat about cluster");
        #     sys.exit(0)
        #
        lock_loop.close()

        # If version was changed from meta server progress dropped for that worker to 0

        config = data["info"]["config"]
        if data["status"] != "LOCK_ACQUIRED":
            version_new = data["info"]["version"]
            self.log.info(f"Version changed from meta server. Worker state: {data['status']} was: {self.version} now: {version_new}")
            self.version = version_new
            up_loop = asyncio.new_event_loop()
            up_loop.run_until_complete(self.update_worker_perc("set_perc", "0"))
            up_loop.close()
        else:
            self.log.info(f"Looks like it new run. Version: {self.version}")

        self.log.info(f"Config getting from meta server: {config}")

        conf = self.load_config("configs/" + config)

        self.searchmap_service_name = self.full_withEnv(conf['settings']['searchmapServiceName'])
        self.service = self.full_withEnv(conf['settings']['service'])
        # producer = full_withEnv(conf['settings']['producer'])
        self.consumer_offset = int(self.full_withEnv(conf['settings']['consumerOffset']))
        self.index_offset = int(self.full_withEnv(conf['settings']['indexOffset']))
        self.backup_offset = int(self.full_withEnv(conf['settings']['backupOffset']))
        self.lucene_shards = int(self.full_withEnv(conf['settings']['luceneShards']))
        self.searchmap_path = self.full_withEnv(conf['settings']['searchmap'])
        self.workers_count = int(self.full_withEnv(conf['settings']['workersCount']))
        self.shard_backup_timeout = int(self.full_withEnv(conf['settings']['shardBackupTimeout']))

        self.perc_per_inum = 0
        self.actual_prog_perc = 0

        searchmap = get_searchmap_parsed(self.searchmap_path)

        service_d = searchmap[self.searchmap_service_name]

        self.stat_dict = {}

        loop = asyncio.get_event_loop()
        # loop = asyncio.get_running_loop()

        # TODO: We need decide when we need update stats about hosts - to backup another host if prev host was halted
        #  actually we get info after we get lock

        for inum, hosts in service_d.items():

            self.stat_dict[inum] = []

            # self.log.info(inum)
            # self.log.info(hosts)
            for hst in hosts:
                hostname_p = f"{hst['hostname']}:{int(hst['search_port']) + self.backup_offset}"
                # stat_dict[inum][hostname_p] = []

                self.log.info(f"Requesting: http://{hst['hostname']}:{int(hst['search_port']) + self.consumer_offset}/status")

                data = loop.run_until_complete(
                    self.gethttp_req(
                        f"http://{hst['hostname']}:{int(hst['search_port']) + self.consumer_offset}/status"))

                self.log.info(f"Requesting: http://{hst['hostname']}:{int(hst['search_port']) + self.index_offset}/status")
                data_i = loop.run_until_complete(
                    self.gethttp_req(f"http://{hst['hostname']}:{int(hst['search_port']) + self.index_offset}/stat"))

                i_halted = data['has-halted-shards']

                i_progress = "False"
                i_empty = "False"

                for metric_i in data_i:
                    if 'index-empty-total' in metric_i[0]:
                        i_empty = metric_i[1]
                    if 'index-copy-progress_perc' in metric_i[0]:
                        i_progress = metric_i[1]

                self.log.info(
                    f"inum: {inum}, host:{hst['hostname']}, halted:{i_halted}, progress:{i_progress}, empty:{i_empty}")

                if i_halted:
                    continue
                if str(i_empty) != "0":
                    continue
                if str(i_progress) != "100.0":
                    continue

                # self.log.info(f"{inum} {hst}")

                lshard_d = {}
                for shrd in range(int(hst['shards'].split("-")[0]), int(hst['shards'].split("-")[1])):
                    lshrd = shrd % self.lucene_shards
                    if not lshard_d.get(lshrd):
                        lshard_d[lshrd] = []
                    # for logging purporses only, actually we can use set here
                    lshard_d[lshrd].append(shrd)

                shard_list = []
                for lshrd in lshard_d.keys():
                    # stat_dict[inum][hst['hostname']].append({'lshard': lshrd, 'shards': lshard_d[lshrd]})
                    # stat_dict[inum][hostname_p].append({'lshard': lshrd})
                    shard_list.append(lshrd)

                self.stat_dict[inum].append({"hostname": hostname_p, "shards": shard_list, "inum": inum})

                self.log.info(self.stat_dict)

                # so in the end we have dict with:
                # {'iNum:0': {'vla2-0479-e68-disksearch-lucene-test-25512.gencfg-c.yandex.net': [{'lshard': 0, 'shards':[30,60...]},{'lshard': 1, 'shards':[30,60...]...}]
                # 'hst2':}}
        loop.close()

    async def getlock(self, f_timeout=20, t_timeout=120):
        # pass_cnt = 0
        # flag = False
        # while pass_cnt <= 20:
        while True:
            w_timeout = randint(f_timeout, t_timeout)
            # Before get lock - increment version to current timestamp
            version = int(time.time())
            self.log.info(f"Try to get lock for version {version}")
            lockurl = f'{LOCK_URL}?hostname={self.hostname}&timestamp={int(time.time())}&version={version}'
            try:
                self.log.info(f"Requesting: {lockurl}")
                data = await self.gethttp_req(lockurl)
                if data["status"] == STATUS_WORKER_TRY_LATER:
                    self.log.info(f"Cannot get lock. Retrying. Trying version to: {version}"
                                  f" Answer: {data} wait {w_timeout} secs and continue")
                    await asyncio.sleep(w_timeout)
                    # pass_cnt += 1
                    continue
                flag = True
                break
            except Exception as e:
                self.log.info(f"Exception reached: {e}, wait {w_timeout} secs and continue")
                # pass_cnt += 1
                await asyncio.sleep(w_timeout)
        # return flag, version, data
        return version, data

    def start(self):
        asyncio.run(self.__start__(self.stat_dict))

    def load_config(self, config):
        with open(config) as f:
            conf = load("".join(f.readlines()), Loader=SafeLoader)
        return conf

    def full_withEnv(self, stringvar):
        stringvar = stringvar % os.environ
        return stringvar

    async def backupRequest(self, post_data, url, inum, session):
        self.log.info(f"INUM: {inum} post data {post_data} url: {url}")
        async with session.post(
            url=url,
            json=post_data

        ) as response:
            return [response]

    async def infoBackupRequest(self,
                                url: str,
                                inum: str,
                                session: aiohttp.ClientSession):
        self.log.info(f"INUM: {inum} get url: {url}")
        async with session.get(url) as r:
            json = {}
            if r.status == 200:
                json = await r.json()
            return r.status, json

    async def sendBackupRequest(self, inum, i_url, b_url, post, shard, shard_perc, stimeout=3600):

        async with aiohttp.ClientSession() as session:
            while True:
                pass_cnt = 0
                try:
                    self.log.info(f"Prepare step. "
                                  f"Renew shard status: {shard} inum: {inum} ")
                    try:
                        code, data_j = await self.infoBackupRequest(i_url, inum, session)
                        if code != 200:
                            self.log.info("Bad meta answer, waiting")
                            await asyncio.sleep(5)
                            continue
                        if data_j['status'] == STATUS_WORKER_FINISHED:
                            self.log.info(f"Shard for this version already in finished state, passing: {shard}, "
                                          f"inum: {inum}, state: {data_j['status']}")
                            # await asyncio.sleep(1)
                            break
                        else:
                            self.log.info(f"Shard for this version not in finished state, backuping: {shard}, "
                                          f"inum: {inum}, state: {data_j['status']}")
                    except Exception as e:
                        self.log.info(
                            f"Backup meta info req error! Retry info request, check state in meta server: {shard} "
                            f"inum: {inum} Error: {e}")
                        await asyncio.sleep(10)
                        continue

                    try:
                        answer = await self.backupRequest(post, b_url, inum, session)
                        self.log.info(f"Backup req answer is {answer}")
                    except Exception as e:
                        # Pass error here - check backup status in meta server and decide - restart backup job
                        self.log.info(
                            f"Backup req error! Pass error, check state in meta server: {shard} inum: {inum} Error: {e}")

                    # TODO: make count_noninit for non initialized if meta server goes down - clients cannot init backups

                    timeout_start = time.time()
                    while True:
                        self.log.info("Renew shard status")

                        try:
                            code, data_j = await self.infoBackupRequest(i_url, inum, session)
                        except Exception as e:
                            self.log.info(
                                f"Backup meta info req error! Retry info request, check state in meta server: {shard} inum: {inum} Error: {e}")
                            await asyncio.sleep(10)
                            continue

                        self.log.info(f"Info api answer status is {code}")
                        self.log.info(f"Retry shard, shard: {shard}, inum: {inum}, state: {data_j['status']}")
                        if time.time() >= timeout_start + stimeout:
                            self.log.info(
                                f"Shard backup timeout reached, retry backup request, shard: {shard}, inum: {inum}, state: {data_j['status']}")
                            raise BackpackWorkerStatusError
                        # self.log.info(f"Info req answer is {data_j}")
                        if code != 200:
                            self.log.info("Bad meta answer, waiting")
                            await asyncio.sleep(5)
                            continue
                        else:
                            # stat = await answer_i.json()
                            # self.log.info(f"FULL STATUS IS {data_j}")
                            if data_j['status'] == STATUS_ERROR:
                                # TODO: make sjme retry before raise exception
                                self.log.info(f"Shard backup failed, shard: {shard}, inum: {inum}, state: {data_j['status']}")
                                raise BackpackWorkerStatusError
                            elif data_j['status'] == STATUS_UNKNOWN:
                                if pass_cnt > 10:
                                    self.log.info(
                                        f"Shard in UNKNOWN state too long restart shard backup shard: {shard}, inum: {inum}, state: {data_j['status']}, pass count {pass_cnt}")
                                    raise BackpackWorkerStatusError
                                else:
                                    self.log.info(
                                        f"Shard in UNKNOWN state, retry info req shard: {shard}, inum: {inum}, state: {data_j['status']}, pass count {pass_cnt}")
                                    await asyncio.sleep(5)
                                    pass_cnt += 1
                                    continue
                            elif data_j['status'] != STATUS_WORKER_FINISHED:
                                self.log.info(f"Waiting shard finished shard: {shard}, inum: {inum}, state: {data_j['status']}")
                                await asyncio.sleep(5)
                                continue
                        self.log.info(
                            f"Shard checks done, all success, shard: {shard}, inum: {inum}, state: {data_j['status']}")
                        break
                except Exception as e:
                    self.log.info(f"Cannot done shard: {shard}, inum: {inum}. Error: {e} Sleep 600 secs, and retry shard.")
                    await asyncio.sleep(600)
                    continue

                self.log.info(f"Shard already backuped, shard: {shard}, inum: {inum}, state: {data_j['status']}")
                break

            self.log.info(f"Shard {shard} inum {inum} backup finished.")

            await self.update_worker_perc("increment_perc", shard_perc)

    def doBackup(self, inum, idata):
        self.log.info(f"Backup inum:{inum}")
        # TODO: rewrite this logic, there we take backup from first valid host only
        #  we need to check host before do backup, maybe
        # self.log.info(f"SHDATA {shdata}")
        hostname = idata[0]['hostname']
        shards = idata[0]['shards']
        i_inum = inum.split(":")[1]

        b_url = f"http://{hostname}/backup/lucene"

        shard_perc = float(self.perc_per_inum) / len(shards)

        for shard in shards:
            i_url = f"{INFO_URL}?version={self.version}&service={self.service}&shard={shard}&inum={i_inum}"

            post = {"version": f"{self.version}",
                    "shard": f"{shard}",
                    "service": f"{self.service}"}
            asyncio.run(self.sendBackupRequest(inum,
                                               i_url,
                                               b_url,
                                               post,
                                               shard,
                                               shard_perc,
                                               stimeout=self.shard_backup_timeout))

    async def __start__(self, stat_dict):
        # TODO: we need separate workers per hosts to avoid overload
        self.perc_per_inum = 100.0 / len(stat_dict.items())
        with ProcessPoolExecutor(max_workers=self.workers_count) as inum_pool:
            # inum_pool = ProcessPoolExecutor(max_workers=self.workers_count)
            tasksToRunInExecutor = []
            for iNum, idata in stat_dict.items():
                # cloop = asyncio.get_event_loop()
                # cloop.call_soon(self.setpercp)
                # cloop.run_in_executor(
                #     inum_pool, self.doBackup, iNum, idata
                # )
                task = asyncio.get_event_loop().run_in_executor(
                    inum_pool,
                    self.doBackup,
                    iNum,
                    idata
                )

                tasksToRunInExecutor.append(
                    task
                )

            # We send all tasks in process executer here and await will be done fast
            num = 0
            for taskToExecute in asyncio.as_completed(tasksToRunInExecutor):
                await taskToExecute
                num += 1
                self.log.info(f"Taskfinished {num}")

            self.log.info("FINISHED BACKUP")
            await self.set_worker_status(STATUS_WORKER_FINISHED)

    async def set_worker_status(self, status):
        slot_up_url = f'{UPDATE_LOCK_URL}?action=set_status&hostname={self.hostname}' \
                      f'&timestamp={int(time.time())}' \
                      f'&version={self.version}' \
                      f'&status={status}'
        try:
            data = await self.gethttp_req(slot_up_url)
            self.log.info(f"Set finished status answer {data}")
        except Exception as e:
            self.log.info(f"Error when try to update slot status {self.hostname} {self.version} {slot_up_url} error {e}")

    async def update_worker_perc(self, action, shard_perc):
        slot_up_url = f'{UPDATE_LOCK_URL}?action={action}&hostname={self.hostname}&timestamp={int(time.time())}' \
                      f'&version={self.version}&percent={shard_perc}'
        try:
            data = await self.gethttp_req(slot_up_url)
            self.log.info(f"Set finished status answer {data}")
        except Exception as e:
            self.log.info(f"Error when try to update slot status {self.hostname} {self.version} {slot_up_url} error {e}")

    async def gethttp_req(self, url):
        async with aiohttp.ClientSession() as session:
            with async_timeout.timeout(10):
                async with session.get(url) as response:
                    return await response.json()


def main():
    parser = OptionParser()

    parser.add_option("-c", "--config", type="string", dest="config",
                      metavar="CONFIG", default="configs/disk_search_backend_test.conf",
                      help="YAML config file (default: %default)")

    (options, args) = parser.parse_args()

    worker = Worker(defaultconfig=options.config)
    worker.start()


if __name__ == '__main__':
    main()
