import json

from aiohttp import web
from yp.client import YpClient
import time
import queue


class YPStat:

    def __init__(self, app):
        self.app = app
        self.log = self.app["alog"]
        self.dc = self.app["dc"]
        self.budget = self.app["budget"]

    async def stat(self, request):
        self.log.info(f'{__name__} called')

        self.log.info(f'Budget is {self.budget}')

        evFilter = '[/meta/pod_set_id] = "%(service)s" ' \
                   'AND ([/status/eviction/state] = "requested" ' \
                   'AND ([/status/eviction/reason] != "hfsm" ' \
                   'OR [/status/maintenance/state] = "requested"))'

        # Get data from yp for instancex in maintenance state
        data_dict = {}
        for service in self.app["service_list"]:
            data_dict[service] = {}
            client = YpClient(self.dc, config={"token": self.app["token"]})
            a_filter = evFilter % {"service": service}

            self.log.info(f'Get hsts by filter: {a_filter} for dc {self.dc}')

            for fields in client.select_objects("pod",
                                                filter=a_filter,
                                                selectors=["/meta/id", "/status/maintenance"]):
                data_dict[service][fields[0]] = fields[1]

        stat_list = []

        states_set = ("none", "requested", "acknowledged", "in_progress")

        # also we use "unknown" state here, for unknown causes
        possibly_broken_causes = {"disk": "Disk check failed",
                                  "memory": "correctable errors during",
                                  "filesystem": "Filesystem is broken",
                                  "host_unreachable": "Host is not available",
                                  "maintenance": "Maintenance reset"}

        state_dict = {}
        cause_dict = {}
        last_updated_dict = {}

        for service, service_data in data_dict.items():
            # write stat: total eviction jobs per service
            stat_list.append([f"{service}-eviction-jobs-total-{self.dc}_axxx", len(service_data)])

            # states eviction
            state_dict[service] = {}

            # eviction causes
            cause_dict[service] = {}

            # timedelta top5 eviction
            # last_updated_dict[service] = []
            last_updated_dict[service] = queue.PriorityQueue(maxsize=0)

            # force set all states to zero
            for state in states_set:
                state_dict[service][state] = 0

            cause_dict[service]["unknown"] = 0
            for p_cause in possibly_broken_causes.keys():
                cause_dict[service][p_cause] = 0

            overall_update_delta = 0
            for instance, i_data in service_data.items():
                # get stat: get state stat
                if i_data["state"] in state_dict[service].keys():
                    state_dict[service][i_data["state"]] += 1
                else:
                    state_dict[service]["unknown"] += 1

                # get stat: possibly causes stat
                flag = False
                for p_cause, p_cause_grep in possibly_broken_causes.items():
                    if p_cause_grep in i_data["info"]["message"]:
                        cause_dict[service][p_cause] += 1
                        flag = True
                        break
                    else:
                        continue

                if not flag:
                    cause_dict[service]["unknown"] += 1

                # Get last update stat
                # we need only five elements - drop smallest, if we have more
                if last_updated_dict[service].qsize() > 5:
                    last_updated_dict[service].get(False)

                update_delta = int(time.time() - float(i_data["last_updated"]) / 1000000)
                last_updated_dict[service].put([update_delta * -1,
                                                instance])

                overall_update_delta += update_delta

            # write stat: overall update delta for all evicted instances
            stat_list.append([f"{service}-eviction-lag-total-{self.dc}_axxx", overall_update_delta])

        # write stat: eviction per state count
        for service, j_state in state_dict.items():
            for c_state, c_count in j_state.items():
                stat_list.append([f"{service}-eviction-state-{c_state}-{self.dc}_axxx", c_count])

        # write stat: possibly causes for evictions
        for service, p_causes in cause_dict.items():
            for p_cause, p_count in p_causes.items():
                stat_list.append([f"{service}-eviction-cause-{p_cause}-{self.dc}_axxx", p_count])

        # write stat: make top-5 update lag pods
        for service, l_update_queue in last_updated_dict.items():
            # we haven't guarantee that we have five element in queue,
            # we need make ghost entries with zero lag
            for i in range(0, 5):
                if l_update_queue.empty():
                    self.log.info("Instance: no_instance Lag: 0")
                    stat_list.append([f"{service}-eviction-lag-{i}-{self.dc}_axxx", 0])
                    continue
                else:
                    elem = l_update_queue.get(False)
                    # TODO: We can add pod number into signal multiplied by 10000 and
                    #  then diff signal into golovan panel.
                    #  But we can show pod number only if we show geo separatly
                    # lag_sec = abs(elem[0]) * 10000
                    # try:
                    #    podnumber = int(elem[1].split("-")[-1])
                    # except Exception:
                    #    podnumber = 0
                    # lag_sec = lag_sec + podnumber
                    # lag_float="{lag_sec}{num:.4f}".format(lag_sec=lag_sec, num=int(elem[1].split("-")[-1])/10000)
                    self.log.info(f'Instance: {elem[1]} Lag: {abs(elem[0])}')
                    # stat_list.append([f"{service}-eviction-lag-{i}-{self.dc}_axxx", lag_sec])
                    stat_list.append([f"{service}-eviction-lag-{i}-{self.dc}_axxx", abs(elem[0])])

        return web.Response(text=json.dumps(stat_list))
