import asyncio

import ssl
import socket

import logging.handlers
import os
from yaml import load, SafeLoader
from optparse import OptionParser

from aiohttp import web,ClientSession
from aiohttp.web_app import Application

from .handlers.Ping import Ping
from .handlers.YPStat import YPStat
from .handlers.GetBudget import GetBudget


import uvloop
asyncio.set_event_loop_policy(uvloop.EventLoopPolicy())


def load_config(config):
    with open(config) as f:
        conf = load("".join(f.readlines()),  Loader=SafeLoader)
    return conf


def full_withEnv(stringvar):
    stringvar = stringvar % os.environ
    return stringvar

async def async_get_stat_job():
    while True:
        print("get_stat_job started")
        await asyncio.sleep(1)
        async with ClientSession() as session:
            async with session.get('http://localhost:8080/makeStat') as resp:
                print(resp.status)
                print(await resp.text())
        print("get_stat_job done")

def main():
    parser = OptionParser()
    parser.add_option("-c", "--config", type="string", dest="config",
                      metavar="CONFIG", default="config_qa.yaml",
                      help="YAML config file (default: %default)")
    (options, args) = parser.parse_args()

    conf = load_config(options.config)

    httpServerPort = int(full_withEnv(conf['server']['httpServerPort']))
    full_log_path = full_withEnv(conf['log']['fullLogPath'])
    access_log_path = full_withEnv(conf['log']['accessLogPath'])
    debug_log_path = full_withEnv(conf['log']['debugLogPath'])
    log_format = conf['log']['format']
    stdout_log = conf['log']['stdout']
    caCertFile = full_withEnv(conf['settings']['caCert'])
    debug = full_withEnv(conf['settings']['debug'])

    token = full_withEnv(conf['settings']['oauth_token'])
    service_list = conf['services']['list']
    dc = full_withEnv(conf['settings']['dc'])

    if stdout_log == "False":
        stdout_log = False

    logging.basicConfig(level=logging.DEBUG,
                        format=log_format)

    full_filehandler = logging.handlers.WatchedFileHandler(full_log_path)
    full_filehandler.setFormatter(logging.Formatter(log_format))

    access_filehandler = logging.handlers.WatchedFileHandler(access_log_path)
    access_filehandler.setFormatter(logging.Formatter(log_format))

    debug_filehandler = logging.handlers.WatchedFileHandler(debug_log_path)
    debug_filehandler.setFormatter(logging.Formatter(log_format))

    access_log = logging.getLogger("aiohttp.access")
    server_log = logging.getLogger("aiohttp.server")

    debug_log = logging.getLogger('asyncio')

    access_log.propagate = stdout_log
    server_log.propagate = stdout_log
    debug_log.propagate = stdout_log

    access_log.addHandler(access_filehandler)
    server_log.addHandler(full_filehandler)
    debug_log.addHandler(debug_filehandler)

    async def init_app():
        """Initialize the application server."""

        ctx = ssl.create_default_context(cafile=caCertFile)
        ctx.check_hostname = False
        ctx.verify_mode = ssl.CERT_NONE

        app: Application = web.Application()

        app['alog'] = access_log
        app['slog'] = server_log
        app['token'] = token
        app['service_list'] = service_list
        app['dc'] = dc
        app['budget'] = []

        app.router.add_route('GET', '/ping', Ping(app).ping)
        app.router.add_route('GET', '/makeStat', GetBudget(app).makeStat)
        app.router.add_route('GET', '/stat', YPStat(app).stat)

        return app

    # Listen on both IPv4 and IPv6
    sock = socket.socket(socket.AF_INET6)
    sock.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEPORT, 1)
    sock.bind(('::', httpServerPort))

    loop = asyncio.get_event_loop()
    # loop = uvloop.new_event_loop()
    # loop.set

    asyncio.get_event_loop().create_task(async_get_stat_job())

    access_log.info(f'debug {debug}')
    #
    if debug == "True":
        loop.set_debug(True)

    inited_app = loop.run_until_complete(init_app())
    web.run_app(inited_app, sock=sock, reuse_address=True, reuse_port=True)


if __name__ == '__main__':
    main()
