import tornado.web
import tornado.gen
import traceback
import copy

from bson.objectid import ObjectId
from libs.context.tools import HashDict

from libs.exceptions import MageAPIResultError
from libs.exceptions import MageAPIServerHardError


class BaseHandler(tornado.web.RequestHandler):
    @tornado.gen.coroutine
    def find_hosts(self, collection, query, limit):
        settings = self.settings["db"]
        result = yield settings.find_cursor(collection, query).to_list(length=limit)
        raise tornado.gen.Return(result)

    @tornado.gen.coroutine
    def bulk_truncate_hosts(self, collection, query, update):
        settings = self.settings["db"]
        result = yield settings.bulk_truncate_cursor(collection, query, update)
        stats = {'deleted_count': result.deleted_count,
                 'inserted_count': result.inserted_count,
                 'matched_count': result.matched_count,
                 'modified_count': result.modified_count,
                 'upserted_count': result.upserted_count
                 }
        raise tornado.gen.Return(stats)

    @tornado.gen.coroutine
    def update_hosts(self, collection, query, update):
        settings = self.settings["db"]
        result = yield settings.update_many_cursor(collection, query, update)
        raise tornado.gen.Return(result)

    @tornado.gen.coroutine
    def syncHosts(self):
        gencfg_service_names = self.generator.get_gencfg_groups(self.generator.NANNY_GROUPS,
                                                                timeout=600,
                                                                oauth_token=self.settings["oauth_token"])

        fat_groups = self.generator.FAT_GROUPS

        hosts_list = self.generator.get_gencfg_instances(gencfg_service_names,
                                                         self.revision,
                                                         timeout=600,
                                                         fat_groups=fat_groups)

        rich_hosts_list = self.enrichHostsList(hosts_list, self.generator.PROJECT)

        update_result = yield self.bulk_truncate_hosts(self.revision,
                                                       {"project": self.generator.PROJECT},
                                                       rich_hosts_list
                                                       )

        self.log.info("Revision:{0} update status is: {1} ".format(self.revision,
                                                                   update_result
                                                                   ))
        self.log.info("Success. Rev:{0} for groups: {1} was synced from GENCFG. Update status is: {2}"
                      .format(self.revision,
                              gencfg_service_names,
                              str(update_result)))

    @tornado.gen.coroutine
    def genRecSplitFatMap(self, recluster=True):
        self.log.info(
            "Generate split searchmap. Searchmap contains all sequences but not contain fat shards sequences.")
        thin_searchmap = yield self.find_hosts(self.searchmap_revision,
                                               {"project": self.generator.PROJECT,
                                                "type": self.generator.MAPTYPE},
                                               limit=100000)

        fat_searchmap = yield self.find_hosts(self.searchmap_revision,
                                              {"project": self.generator.PROJECT,
                                               "type": self.generator.FATMAPTYPE},
                                              limit=100000)

        # Get baseshards from fat shardmap per service
        fat_shardlist = {}
        for line in fat_searchmap:
            if not fat_shardlist.get(line["service"]):
                fat_shardlist[line["service"]] = set()
            fat_shardlist[line["service"]].add(line["shards"].split("-")[0])

        # we need to generate splitmap for all services
        split_thinmap = self.generate_split_thinmap(thin_searchmap,
                                                    fat_shardlist)

        #print split_thinmap

        #for line in split_thinmap:
        #    print line

        yield self.writemap(split_thinmap,
                            revision=self.searchmap_split_revision,
                            maptype=self.generator.MAPTYPE)
        
        yield self.writemap(fat_searchmap,
                            revision=self.searchmap_split_revision,
                            maptype=self.generator.FATMAPTYPE)

        if recluster:
            self.log.info(
                "Generate recluster split searchmap. Searchmap contains all sequences but not contain fat shards "
                "sequences.")
            recluster_thin_searchmap = yield self.find_hosts(self.recluster_searchmap_revision,
                                                             {"project": self.generator.PROJECT,
                                                              "type": self.generator.MAPTYPE},
                                                             limit=100000)

            recluster_split_thinmap = self.generate_split_thinmap(recluster_thin_searchmap,
                                                                  fat_shardlist)

            yield self.writemap(recluster_split_thinmap,
                                revision=self.recluster_searchmap_split_revision,
                                maptype=self.generator.MAPTYPE)

            # write from one collection to other collection
            recluster_fat_searchmap = yield self.find_hosts(self.recluster_searchmap_revision,
                                                            {"project": self.generator.PROJECT,
                                                             "type": self.generator.FATMAPTYPE},
                                                            limit=100000)

            yield self.writemap(recluster_fat_searchmap,
                                revision=self.recluster_searchmap_split_revision,
                               maptype=self.generator.FATMAPTYPE)

    @staticmethod
    def enrichHostsList(hosts_list, project):
        for host in hosts_list:
            host['tag'] = "{0}_{1}".format(host['hostname'].split(".")[0],
                                           host['port'])

            # We need tag_mtn for Syncmap action - to renew old searchmap.
            if host["mtn_backbone_hostname"] != "False":
                host['tag_mtn'] = host["mtn_backbone_hostname"].split(".")[0]
            else:
                host['tag_mtn'] = "False"
            host['project'] = project
        return hosts_list

    def generate_split_thinmap(self, thin_searchmap, fatshardlist):

        partialmap, partialmap_fat = self.split_thinmap(thin_searchmap, fatshardlist)

        # TODO: make test for shard splitting before any updates
        changed_map = self.wo_fatshards_map(partialmap_fat, fatshardlist)
        split_thin_searchmap = partialmap + changed_map

        return split_thin_searchmap

    @staticmethod
    def split_thinmap(thin_searchmap, fatshards):
        # TODO: Doubled?
        # function receives two lists - with searchmap hosts, and with fat shards numbers
        # and return two lists - list with hosts not contains fatshards (partialmap), and list with hosts contains
        # fat shard (partialmap_fat)
        partialmap = []
        partialmap_fat = set()
        for service, shardlist in fatshards.iteritems():
            for hst in thin_searchmap:
                # if two fat shards hit in same shard range - only one line must be selected
                check_shard = {}
                flag = False
                if hst["service"] == service:
                    check_shard[hst["shards"]] = False
                    hst_shards = range(int(hst["shards"].split("-")[0]), int(hst["shards"].split("-")[1]) + 1)
                    for fatshard in shardlist:
                        if int(fatshard) in hst_shards:
                            flag = True
                            if not check_shard[hst["shards"]]:
                                tmp_hst = copy.deepcopy(hst)
                                tmp_hst["_id"] = ObjectId()
                                partialmap_fat.add(HashDict(tmp_hst))
                                check_shard[hst["shards"]] = True
                    if not flag:
                        partialmap.append(hst)
        return partialmap, partialmap_fat

        #for hst in thin_searchmap:
        #    # u'shards': u'0-30'
        #    hst_shards = range(int(hst["shards"].split("-")[0]), int(hst["shards"].split("-")[1]) + 1)
        #    flag = False
        #    for fatshard in fatshards[hst["service"]]:
        #        # print fatshard
        #        if int(fatshard) in hst_shards:
        #            # So, we know that our fat shard in this range
        #            # We need split this line to consecutives ranges
        #            # Now - copy to partialmap_fat set
        #            tmp_hst = copy.deepcopy(hst)
        #            tmp_hst["_id"] = ObjectId()
        #            partialmap_fat.add(HashDict(tmp_hst))
        #            flag = True
        #    if not flag:
        #        partialmap.append(hst)
        #return partialmap, partialmap_fat

    @staticmethod
    def wo_fatshards_map(fullmap_fat, fatshards):
        # function cut fat shard from shards range for host in hostlist
        # and return list with hosts with all consecutives ranges for shard
        changed_map = []
        for hst in fullmap_fat:
            hst_shards = set(range(int(hst["shards"].split("-")[0]), int(hst["shards"].split("-")[1]) + 1))
            for fatshard in fatshards[hst["service"]]:
                hst_shards = hst_shards - set([int(fatshard)])

            shrd_groups = BaseHandler.group_consecutives(sorted(list(hst_shards)))
            for shrd_group in shrd_groups:
                hst_temp = copy.deepcopy(dict(hst))
                hst_temp["_id"] = ObjectId()
                hst_temp["shards"] = "{0}-{1}".format(shrd_group[0], shrd_group[-1])
                changed_map.append(hst_temp)
        return changed_map

    @staticmethod
    def group_consecutives(vals, step=1):
        """Return list of consecutive lists of numbers from vals (number list)."""
        run = []
        result = [run]
        expect = None
        for v in vals:
            if (v == expect) or (expect is None):
                run.append(v)
            else:
                run = [v]
                result.append(run)
            expect = v + step
        return result

    @tornado.gen.coroutine
    def checkUpdateHosts(self):
        # Check all hosts and sync all host (including fat)
        curr_hostlist = yield self.getHosts(hosttype=False, limit=1000)

        if not curr_hostlist or self.resync == "true":
            self.log.info("Cannot found hosts for generation or resync specified. "
                          "Resync and overwrite gencfg groups")
            # sync all gropus for services including fat
            yield self.syncHosts()
        else:
            self.log.info("Resync param NOT specified. Use previosly synced gencfg groups for map. ")

        #raise tornado.gen.Return(curr_hostlist)

    @tornado.gen.coroutine
    def getHosts(self, hosttype=False, limit=60000):
        qs = {"project": self.generator.PROJECT}

        if hosttype:
            qs["type"] = hosttype

        curr_hostlist = yield self.find_hosts(self.revision,
                                              qs,
                                              limit=limit)
        raise tornado.gen.Return(curr_hostlist)


    @tornado.gen.coroutine
    def writemap(self, data, revision=None, maptype=None):
        if not maptype or not revision:
            raise MageAPIServerHardError("Maptype or revison does't specified.")

        self.log.info(
            "Revision: {0} Project: {1} Type: {2} generation complete. Updating records in database.".format(revision,
                                                                                                             self.generator.PROJECT,
                                                                                                             maptype))

        update_result = yield self.bulk_truncate_hosts(revision,
                                                       {"project": self.generator.PROJECT,
                                                        "type": maptype},
                                                       data
                                                       )

        self.log.info("Revision: {0} Project: {1} Type: {2} searchmap update stats: {3}".format(revision,
                                                                                                self.generator.PROJECT,
                                                                                                maptype,
                                                                                                update_result))

        self.write("\nRevision: {0} Project: {1} Type: {2} searchmap update stats: {3}\n".format(revision,
                                                                                               self.generator.PROJECT,
                                                                                               maptype,
                                                                                               update_result))

    def _on_response(self, result, error, code=501):
        if error:
            if isinstance(error, str):
                raise tornado.web.HTTPError(code, error)
            else:
                raise tornado.web.HTTPError(code, str(error.details))
        else:
            self.write(str(result))
        self.finish()

    def _on_response_write_map(self, result, error):
        if error:
            raise tornado.web.HTTPError(500, log_message=error)
        else:
            self.write_searchmap_to_output(result, [])
        self.finish()

    def write_searchmap_to_output(self, result, new_format_groups):
        try:
            if self.type != "fat":
                if len(result) < self.generator.MINMAPSIZE and self.full:
                    raise MageAPIResultError(500, "Searchmap is too small!")
            for el in result:
                #if el['group'] in new_format_groups:
                #    if not self.generator.FMT_NEW.get(el['service'], False):
                #        line = self.generator.FMT_NEW['default'] % el
                #    else:
                #        line = self.generator.FMT_NEW[el['service']] % el
                #else:
                #    if not self.generator.FMT.get(el['service'], False):
                #        line = self.generator.FMT['default'] % el
                #    else:
                #        line = self.generator.FMT[el['service']] % el
                if self.pformat:
                    if not self.generator.FMT.get("{0}_{1}".format(self.pformat, el['service']), False):
                        line = self.generator.FMT["{0}_{1}".format(self.pformat, "default")] % el
                    else:
                        line = self.generator.FMT["{0}_{1}".format(self.pformat, el['service'])] % el
                else:
                    if not self.generator.FMT.get(el['service'], False):
                        line = self.generator.FMT['default'] % el
                    else:
                        line = self.generator.FMT[el['service']] % el
                self.write(str("\n"))
                self.write(str(line))
            self.write(str("\n"))
        except Exception as e:
            self.log.debug("Error reached when write to output: {0} {1}".format(e.message, e))
            raise MageAPIServerHardError("Exception occured:{0} {1}".format(e, traceback.print_exc()))