import tornado.gen

from libs.exceptions import MageAPIServerHardError
from BaseHandler import BaseHandler


class GenmapHandler(BaseHandler):
    """Generate simple single searchmap"""

    def __init__(self, application, request, **kwargs):
        super(GenmapHandler, self).__init__(application, request, **kwargs)
        self.revision = self.get_argument("revision")
        self.searchmap_revision = "{0}:{1}".format(self.generator.SEARCHMAP_PREFIX, self.revision)

        self.resync = self.get_argument("resync", False)
        self.fatonly = self.get_argument("fatonly", False)
        self.force = self.get_argument("force", False)

    def initialize(self, generator):
        self.generator = generator
        self.log = generator.log
        self.log.info("Trying to generate map for project: {0}".format(generator.PROJECT))

    @tornado.gen.coroutine
    def get(self):
        # Update including fat too
        yield self.checkUpdateHosts()

        update_result = {}

        # TODO: check force flag
        if not self.fatonly:
            currthin = yield self.find_hosts(self.searchmap_revision,
                                             {"project": self.generator.PROJECT, "type": self.generator.MAPTYPE},
                                             limit=100000)
            if currthin and not self.force == "yes_i_want_this_uguu":
                raise MageAPIServerHardError("Current searchmap already founded in database. "
                                             "If you know what are you doing - and you want to owerwrite it, "
                                             "add argument \"force=yes_i_want_this_uguu\". "
                                             "Revision:{0} Type: {1}"
                                             .format(self.searchmap_revision, self.generator.MAPTYPE))
            update_result = yield self.genThinMap()
            self.write("Update collection: {0} Type: {1} Result: {2}\n".format(self.searchmap_revision,
                                                                               self.generator.MAPTYPE,
                                                                               str(update_result)))
        else:
            self.log.info("Fatonly option reached. Genmap for fat groups only if they exists.")

        if self.generator.ISFAT:
            self.searchmap_split_revision = "{0}:{1}".format(self.generator.SEARCHMAP_SPLIT_PREFIX, self.revision)

            currfat = yield self.find_hosts(self.searchmap_revision,
                                            {"project": self.generator.PROJECT, "type": self.generator.FATMAPTYPE},
                                            limit=10000)

            if currfat and not self.force == "yes_i_want_this_uguu":
                raise MageAPIServerHardError("Current searchmap already founded in database. "
                                             "If you know what are you doing - and you want to owerwrite it, "
                                             "add argument \"force=yes_i_want_this_uguu\". "
                                             "Revision:{0} Type: {1}"
                                             .format(self.searchmap_revision, self.generator.FATMAPTYPE))

            #TODO: Do we need store fat hosts in simple searchmap?
            update_result = yield self.genFatMap()
            self.write("Update collection: {0} Type: {1} Result: {2}\n".format(self.searchmap_revision,
                                                                               self.generator.FATMAPTYPE,
                                                                               str(update_result)))
            yield self.genRecSplitFatMap(recluster=False)

        self._on_response("Map generation completed successfully. Stats: {0}\n".format(update_result), False)

    @tornado.gen.coroutine
    def genThinMap(self, limit=60000):
        curr_hostlist = yield self.getHosts(hosttype=self.generator.MAPTYPE, limit=limit)

        searchmap = self.generator.gen_singlemap(curr_hostlist,
                                                 oauth_token=self.settings["oauth_token"])

        update_result = yield self.bulk_truncate_hosts(self.searchmap_revision,
                                                       {"project": self.generator.PROJECT,
                                                        "type": self.generator.MAPTYPE},
                                                       searchmap)
        raise tornado.gen.Return(update_result)

    @tornado.gen.coroutine
    def genFatMap(self, limit=1000):
        curr_hostlist = yield self.getHosts(hosttype=self.generator.FATMAPTYPE, limit=limit)

        searchmap = self.generator.gen_fat_singlemap(curr_hostlist,
                                                     oauth_token=self.settings["oauth_token"])

        update_result = yield self.bulk_truncate_hosts(self.searchmap_revision,
                                                       {"project": self.generator.PROJECT,
                                                        "type": self.generator.FATMAPTYPE},
                                                       searchmap)
        raise tornado.gen.Return(update_result)


    #@tornado.gen.coroutine
    #def genFatMap(self):
    #    self.prev_revision = self.get_argument("prev_revision", False)
    #
    #    self.searchmap_prev_revision = "{0}:{1}".format("searchmap", self.prev_revision)
    #
    #    self.searchmap_revision = "{0}:{1}".format("searchmap", self.revision)
    #    self.searchmap_revision_split = "{0}:{1}".format("searchmap_split", self.revision)
    #
    #    self.recluster_searchmap_revision = "{0}:{1}".format("recluster_searchmap", self.revision)
    #    self.recluster_searchmap_revision_split = "{0}:{1}".format("recluster_searchmap_split", self.revision)
    #
    #    prev_fat_searchmap = yield self.find_hosts(self.searchmap_prev_revision,
    #                                               {"project": self.project, "type": "fat"},
    #                                               limit=1000)
    #
    #    curr_fat_hostlist = yield self.find_hosts(self.revision,
    #                                              {"project": self.project, "type": "fat"},
    #                                              limit=1000)
