import tornado.gen

from libs.exceptions import MageAPIServerHardError
from BaseHandler import BaseHandler


class GenmapReclusterHandler(BaseHandler):
    """Generate recluster and searchmap for next revision"""

    def __init__(self, application, request, **kwargs):
        super(GenmapReclusterHandler, self).__init__(application, request, **kwargs)
        self.force = self.get_argument("force", False)
        self.resync = self.get_argument("resync", False)
        self.revision = self.get_argument("revision")
        self.prev_revision = self.get_argument("prev_revision", False)

        self.searchmap_revision = "{0}:{1}".format(self.generator.SEARCHMAP_PREFIX, self.revision)
        self.searchmap_prev_revision = "{0}:{1}".format(self.generator.SEARCHMAP_PREFIX, self.prev_revision)
        self.recluster_searchmap_revision = "{0}:{1}".format(self.generator.RECLUSTER_SEARCHMAP_PREFIX, self.revision)

    def initialize(self, generator):
        self.generator = generator
        self.log = generator.log
        self.log.info("Trying to generate map for project: {0}".format(self.generator.PROJECT))

    @tornado.gen.coroutine
    def get(self):
        yield self.genRecThinMap()
        if self.generator.ISFAT:
            yield self.genRecFatMap()
            self.searchmap_split_revision = "{0}:{1}".format(self.generator.SEARCHMAP_SPLIT_PREFIX, self.revision)
            self.recluster_searchmap_split_revision = "{0}:{1}".format(self.generator.RECLUSTER_SEARCHMAP_SPLIT_PREFIX,
                                                                       self.revision)

            yield self.genRecSplitFatMap()

        self._on_response("\nSuccess. Recluster was finished", False)

    @tornado.gen.coroutine
    def genRecThinMap(self):
        curr_hostlist, prev_searchmap = yield self.checkAndGetMap(maptype=self.generator.MAPTYPE, limit=1000000)
        current_result_searchmap, current_result_recluster_searchmap = self.generator.gen_maps(curr_hostlist,
                                                                                               prev_searchmap,
                                                                                               oauth_token=self.settings["oauth_token"])

        yield self.writemap(current_result_searchmap,
                            revision=self.searchmap_revision,
                            maptype=self.generator.MAPTYPE)

        yield self.writemap(current_result_recluster_searchmap,
                            revision=self.recluster_searchmap_revision,
                            maptype=self.generator.MAPTYPE)

    @tornado.gen.coroutine
    def genRecFatMap(self):
        curr_hostlist, prev_searchmap = yield self.checkAndGetMap(maptype=self.generator.FATMAPTYPE, limit=1000000)
        current_result_fatsearchmap, current_result_recluster_fatsearchmap = self.generator.gen_fat_maps(curr_hostlist,
                                                                                                         prev_searchmap,
                                                                                                         oauth_token=self.settings["oauth_token"])

        yield self.writemap(current_result_fatsearchmap,
                            revision=self.searchmap_revision,
                            maptype=self.generator.FATMAPTYPE)

        yield self.writemap(current_result_recluster_fatsearchmap,
                            revision=self.recluster_searchmap_revision,
                            maptype=self.generator.FATMAPTYPE)

    @tornado.gen.coroutine
    def checkAndGetMap(self, maptype=None, limit=1000):
        prev_searchmap = yield self.find_hosts(self.searchmap_prev_revision,
                                               {"project": self.generator.PROJECT, "type": maptype},
                                               limit=limit)

        curr_searchmap = yield self.find_hosts(self.searchmap_revision,
                                               {"project": self.generator.PROJECT, "type": maptype},
                                               limit=limit)

        if curr_searchmap and not self.force == "yes_i_want_this_uguu":
            self.log.info("Searchmap already founded in database, and force \"!= yes_i_want_this_uguu\" argument not specified."
                          "Exiting with error. Revision:{0} Type: {1}".format(self.searchmap_revision, maptype))
            raise MageAPIServerHardError("Current searchmap already founded in database. "
                                         "If you know what are you doing - and you want to owerwrite it, add argument \"force=yes_i_want_this_uguu\". "
                                         "Revision:{0} Type: {1}"
                                         .format(self.searchmap_revision, maptype))
        elif self.force == "yes_i_want_this_uguu":
            self.log.info("Searchmap already founded in database, but \"force = yes_i_want_this_uguu\" argument exists."
                          " Overwrite existing searchmaps. Revision:{0} Type: {1}".format(self.searchmap_revision, maptype))
            self.write("\nOverwriting founded recluster and searchmap for revision:{0} Type: {1}".format(self.revision, maptype))

        if not prev_searchmap:
            raise MageAPIServerHardError("Cannot find prev searchmap for gen recluster. You need upload (or gen) it firstly. "
                                         "Revision: {0} Type: {1}"
                                         .format(self.searchmap_prev_revision, maptype))
        else:
            self.log.info("Prev searchmap founded. For recluster we need find hosts for revision: {0}. Type: {1}\n"
                          "Try to do it.".
                          format(self.revision, maptype))

            # TODO: All types synced. Shall we sync maptypes separately?
            yield self.checkUpdateHosts()

        curr_hostlist = yield self.getHosts(hosttype=maptype)

        raise tornado.gen.Return((curr_hostlist, prev_searchmap))
