import logging
import tools

import urllib2
import json
import copy
import ssl
import time

from tools import HashDict
from bson.objectid import ObjectId

from libs.exceptions import *

ssl._create_default_https_context = ssl._create_unverified_context


class Base(object):
    log = logging.getLogger("tornado.application")
    NANNY_URL = "https://nanny.yandex-team.ru/v2/services/"
    GENCFG_URL = "http://api.gencfg.yandex-team.ru/"
    TRIES = 3
    SLEEP = 0.5
    MTN_HOSTNAMES_ERROR_LIMIT = 1

    info_fields_hostlist = ['hostname',
                            'port',
                            'group',
                            'mtn_backbone_hostname',
                            'dc'
                            ]

    hashing_map_fields = ['port',
                          'hostname']

    def __init__(self):
        pass

    @staticmethod
    def get_ranges_intersection(r1, r2):
        """
            Get intersection of couple of segments
            :param r1: (tuple) first segment in form [begin, end]
            :param r2: (tuple) second segment in form [begin, end]
            :return: None if no intersection, tuple in form [begin, end] otherwise
        """

        f1, l1 = r1
        f2, l2 = r2

        f_result = max(f1, f2)
        l_result = min(l1, l2)

        if l_result < f_result:
            return None

        return f_result, l_result

    @staticmethod
    def get_lucene_shard_range(shard_count, shard_id, total_shards):

        first_lucene_shard = shard_id * shard_count / total_shards
        last_lucene_shard = (shard_id + 1) * shard_count / total_shards - 1

        return first_lucene_shard, last_lucene_shard

    @staticmethod
    def parsed_loc(hostlist):
        splitted_hosts = {}
        for element in hostlist:
            if not splitted_hosts.get(element['group'], False):
                splitted_hosts[element['group']] = []
            splitted_hosts[element['group']] += [element]
        return splitted_hosts

    @staticmethod
    def make_auth_req(url, token=False, timeout=60):
        """
        TODO: make async http calld
        """

        request = urllib2.Request(url)
        if token:
            request.add_header("Authorization", "OAuth {0}".format(token))
        attempts = 0
        while attempts < Base.TRIES:
            try:
                result = urllib2.urlopen(request, timeout=timeout)
                break
            except Exception as e:
                attempts += 1
                Base.log.info("Error reached {0} Url: {2} Retry count:{1}".format(e, attempts, url))
                if attempts == Base.TRIES:
                    Base.log.info("Timeout!")
                    raise Exception("Timeout exception!")
                time.sleep(Base.SLEEP)
        return result.read()

    @staticmethod
    def get_gencfg_groups(service_list, timeout=20, oauth_token=False):
        g_groups = {}
        for serviceName in service_list:
            g_groups[serviceName] = set()
            data = Base.make_auth_req("{0}{1}".format(Base.NANNY_URL, serviceName), oauth_token, timeout=timeout)
            service_dict = json.loads(data)
            for groupObj in service_dict['runtime_attrs']['content']['instances']['extended_gencfg_groups']['groups']:
                g_groups[serviceName].add(groupObj['name'])
        return g_groups

    # Extended getter - with
    @staticmethod
    def get_gencfg_groups_ext(service_list, timeout=20, oauth_token=False):
        g_groups = {}
        for serviceName in service_list:
            g_groups[serviceName] = {}
            data = Base.make_auth_req("{0}{1}".format(Base.NANNY_URL, serviceName), oauth_token, timeout=timeout)
            service_dict = json.loads(data)
            g_groups[serviceName] = []
            for groupObj in service_dict['runtime_attrs']['content']['instances']['extended_gencfg_groups']['groups']:
                g_groups[serviceName].append({"release": groupObj['release'],
                                              "name": groupObj['name']})
        return g_groups

    @staticmethod
    @tools.lru_cache(log, maxsize=40, timeout=300)
    def get_gencfg_instances(g_groups, revision, timeout=20, fat_groups=None):
        """
        Get json dict with instances from gencfg.
        Url example: http://api.gencfg.yandex-team.ru/unstable/groups/MSK_RESERVED/instances

        :param fat_groups: list fatgroup names, may be empty
        :param g_groups: List of gencfg groups
        :param g_groups: Dict of gencfg groups e.g ({"mail_search_prestable": ["MSK_RESERVERD","VLA_LUCENE"])
        :param revision: Service revision in Gencfg
        :param timeout: timeout
        """
        if not isinstance(fat_groups, dict):
            raise MageAPIServerHardError("You must define fat groups for gencfg instance getter. "
                                         "This list may be empty.")

        def hostlist_locgroup(g_group, service=None):
            result = []
            mtn_hsts_err_c = 0
            for loc_group in g_group:
                Base.log.info("{0}{1}/searcherlookup/groups/{2}/instances".format(Base.GENCFG_URL, revision, loc_group))
                data = Base.make_auth_req("{0}{1}/searcherlookup/groups/{2}/instances".format(Base.GENCFG_URL, revision, loc_group),
                                          timeout=timeout)
                dataj = json.loads(data)['instances']
                for item in dataj:
                    if loc_group in fat_groups.keys():
                        in_type = "fat"
                    else:
                        in_type = "thin"

                    if item["hbf"]["interfaces"].get("backbone", False):
                        mtn_h_name = item["hbf"]["interfaces"]["backbone"]["hostname"]
                        mtn_addr = item["hbf"]["interfaces"]["backbone"]["ipv6addr"]
                    else:
                        Base.log.info("WARNING! Cannot get mtn hostname from GENCFG for hostname: {0} Error count increased."
                                      "Current error count: {1}".format(item["hostname"],mtn_hsts_err_c))
                        mtn_h_name = "False"
                        mtn_addr = "False"
                        mtn_hsts_err_c += 1

                    to_upd_item = {"group": loc_group,
                                   "service": service,
                                   "type": in_type,
                                   "dc": item["dc"],
                                   "domain": item["domain"],
                                   "hostname": item["hostname"],
                                   "location": item["location"],
                                   "port": item["port"],
                                   "power": item["power"],
                                   "mtn_backbone_hostname": mtn_h_name,
                                   "mtn_backbone_ipv6addr": mtn_addr
                                   }

                    result.append(copy.deepcopy(to_upd_item))
                if mtn_hsts_err_c > Base.MTN_HOSTNAMES_ERROR_LIMIT:
                    Base.log.info("ERROR! Cannot get mtn hostname from GENCFG for hostname: {0} "
                                  "Error count limit reached. Current error count: {1} "
                                  "Errors limit: {2}".format(item["hostname"],
                                                             mtn_hsts_err_c,
                                                             Base.MTN_HOSTNAMES_ERROR_LIMIT
                                                             ))

                    raise MageAPIError("Reached MTN_HOSTNAMES_ERROR_LIMIT for mtn instances without hostnames. Curr error hosts: {0} Exiting. For details: RTCSUPPORT-974".format(mtn_hsts_err_c))
            return result

        result = []
        if isinstance(g_groups, dict):
            for service, g_group in g_groups.iteritems():
                result = hostlist_locgroup(g_group, service=service) + result
        if isinstance(g_groups, list):
            result = hostlist_locgroup(g_groups)

        return result

    @staticmethod
    def zoo_split_uncover(shardsplit, fat_groups, oauth_token=False):
        mail_split_hosts = {}
        mail_split_hosts_mtn = {}

        #for shard_range, nanny_service_name in shardsplit:
        for queuedata in shardsplit:
            nanny_service_name = queuedata[1]
            shard_range = queuedata[0]
            default_zoo_string = False
            if len(queuedata) == 2:
                zoo_base_port = "False"
            if len(queuedata) == 3:
                zoo_base_port = int(queuedata[2])
                Base.log.info("Zoo base port overrided, queue: {0} port: {1}".format(nanny_service_name, zoo_base_port))
            if len(queuedata) == 4:
                zoo_base_port = int(queuedata[2])
                Base.log.info("Zoo base port overrided, queue: {0} port: {1}".format(nanny_service_name, zoo_base_port))
                default_zoo_string = "|".join(queuedata[3])
                Base.log.info("Zoo queue string was overridden, nanny: {0} queue string: {1}".format(nanny_service_name,
                                                                                                     default_zoo_string))

            # if queue string was overriden - just continue
            if default_zoo_string:
                mail_split_hosts[tuple(shard_range)] = default_zoo_string
                mail_split_hosts_mtn[tuple(shard_range)] = default_zoo_string
                continue

            g_groups = Base.get_gencfg_groups_ext([nanny_service_name], oauth_token=oauth_token)
            Base.log.info("Try to expand groups: {0}".format(g_groups))
            # because there may be two or more group for service
            zoo_string_list = []
            zoo_string_list_mtn = []
            for n_servicen, data in g_groups.iteritems():
                for g_group in data:
                    instances_list = Base.get_gencfg_instances({n_servicen: [g_group['name']]},
                                                               g_group['release'],
                                                               fat_groups=fat_groups)
                    for inst_data in instances_list:
                        inst_portnum = inst_data['port']
                        if isinstance(zoo_base_port, int):
                            inst_portnum = zoo_base_port
                        zoo_string_list.append("{0}:{1}/{2}".format(inst_data['hostname'],
                                                                    inst_portnum,
                                                                    str(inst_portnum + 1)
                                                                    ))

                        zoo_string_list_mtn.append("{0}:{1}/{2}".format(inst_data['mtn_backbone_hostname'],
                                                                        inst_portnum,
                                                                        str(inst_portnum + 1)
                                                                        ))
            zoo_string = "|".join(sorted(zoo_string_list))
            zoo_string_mtn = "|".join(sorted(zoo_string_list_mtn))

            # Use tuple here because tuple is hashable - and we need get zoostring by key, for mtn later
            mail_split_hosts[tuple(shard_range)] = zoo_string
            mail_split_hosts_mtn[tuple(shard_range)] = zoo_string_mtn

        return {"zk": mail_split_hosts, "zk_mtn": mail_split_hosts_mtn}


    def genmap(self,
               base_port,
               indexer_port_offset,
               search_port_offset,
               queue_id_port_offset,
               dump_port_offset,
               offset,
               oldstyle,
               service,
               servicedict,
               replicafactor,
               project,
               zoo_split_uncovered,
               ):
        #byshard_inst_dict = servicedict

        shardmap_result = []

        for shard_id in range(self.INUM_START, self.INUM_END):
            # Dont pass exception here, if any
            instance_group = servicedict[shard_id]
            inst_data = Base.gen_for_shard(self.INUM_TOTAL,
                                           shard_id,
                                           instance_group,
                                           indexer_port_offset,
                                           search_port_offset,
                                           queue_id_port_offset,
                                           dump_port_offset,
                                           offset,
                                           service,
                                           project,
                                           zoo_split_uncovered,
                                           self.SHARD_COUNT,
                                           mapType=self.MAPTYPE,
                                           base_port=base_port
                                           )

            if len(inst_data) != replicafactor:
                # self.log.info("The number of replicas for shard {0} is higher than replicafactor {1} Current: {2}"
                #               .format(shard_id, self.replicafactor, len(inst_data)))
                pass

            shardmap_result += inst_data

        # self.log.info("Total shards: {0}".format(total_shards))
        return shardmap_result

    @staticmethod
    def gen_for_shard(inum_count,
                      shard_id,
                      instance_group,
                      indexer_port_offset,
                      search_port_offset,
                      queue_id_port_offset,
                      dump_port_offset,
                      offset,
                      service,
                      project,
                      zoo_split_uncovered,
                      shard_count,
                      base_port=False,
                      mapType=False,
                      inumFat=False
                      ):

        result = []

        for instance in instance_group:
            if instance["mtn_backbone_hostname"] != "False":
                tag_mtn = instance["mtn_backbone_hostname"].split(".")[0]
            else:
                tag_mtn = "False"

            inst_portnum = int(instance['port'])

            if base_port:
                inst_portnum = base_port

            data = {
                "service": service,
                "iNum": "{0}".format(shard_id),
                "tag": "{0}_{1}".format(instance['hostname'].partition(".")[0], inst_portnum),
                "hostname": instance['hostname'],
                "json_indexer_port": "{0}".format(int(inst_portnum) + indexer_port_offset),
                "search_port": "{0}".format(int(inst_portnum) + search_port_offset),
                "port": str(inst_portnum),
                "queue_id_port": "{0}".format(int(inst_portnum) + queue_id_port_offset),
                "dump_port": "{0}".format(int(inst_portnum) + dump_port_offset),
                "search_port_ng": "{0}".format(int(inst_portnum) + 1),
                "project": project,
                "group": instance['group'],
                "type": mapType,
                "mtn_backbone_hostname": instance['mtn_backbone_hostname'],
                "tag_mtn": tag_mtn,
                "dc": instance['dc']
            }

            if inumFat:
                # we don't need offset becase all fat shards well be placed to the same hosts
                #shard_id_patched = (shard_id + offset) % shard_count
                data["iNum"] = inumFat
                data["shards"] = "%s-%s" % (str(shard_id), str(shard_id))
                data["zk"] = Base.getzkconfig(shard_id, zoo_split_uncovered["zk"])
                data["zk_mtn"] = Base.getzkconfig(shard_id, zoo_split_uncovered["zk_mtn"])
                data["_id"] = ObjectId()
                result.append(data.copy())
            else:
                lucene_first_shard, lucene_last_shard = Base.get_lucene_shard_range(shard_count,
                                                                                    shard_id,
                                                                                    inum_count)
                lucene_first_shard = (lucene_first_shard + offset) % shard_count
                lucene_last_shard = (lucene_last_shard + offset) % shard_count
                lucene_shard_ranges = []
                if lucene_first_shard > lucene_last_shard:
                    lucene_shard_ranges.append((lucene_first_shard, shard_count - 1))
                    lucene_shard_ranges.append((0, lucene_last_shard))
                else:
                    lucene_shard_ranges.append((lucene_first_shard, lucene_last_shard))

                for lucene_first_shard, lucene_last_shard in lucene_shard_ranges:
                    for lucene_split_range, lucene_queue_line in zoo_split_uncovered['zk'].iteritems():
                        r = Base.get_ranges_intersection(lucene_split_range, (lucene_first_shard, lucene_last_shard))
                        if r is not None:
                            data["shards"] = "%s-%s" % (r[0], r[1])
                            data["zk"] = lucene_queue_line
                            # Save zk and zk_mtn line
                            data["zk_mtn"] = zoo_split_uncovered['zk_mtn'][lucene_split_range]
                            data["_id"] = ObjectId()
                            result.append(data.copy())
        return result


    def make_recluster_dict(self, curr_hostlist, prev_searchmap, replicafactor, log, fat=False):
        # the function receive prev searchmap and current gencfg hostlist.
        # return two dicts - with current recluster searchmap, and current searchmap (w/o reclsuter instances)

        prev_tagset = Base.make_location_set(prev_searchmap)
        curr_tagset = Base.make_location_set(curr_hostlist)

        for location, hostset in prev_tagset.iteritems():
            log.info("All hosts from prev searchmap. Location: {0} Len: {1}".format(location, len(hostset)))

        for location, hostset in curr_tagset.iteritems():
            log.info("Hosts from current gencfg revision. Location: {0} Len: {1}".format(location, len(hostset)))

        add_in_curr_tags_locset = Base.set_diff(curr_tagset, prev_tagset)
        removed_in_curr_tags_locset = Base.set_diff(prev_tagset, curr_tagset)

        for location, hostset in add_in_curr_tags_locset.iteritems():
            log.info("Added hosts. Location:{0} Len:{1}".format(location, len(hostset)))

        for location, hostset in removed_in_curr_tags_locset.iteritems():
            log.info("Removed hosts. Location:{0} Len:{1}".format(location, len(hostset)))

        # TODO: Inum FIX - must be str in prev searchmap! (Why i specify it as int?)
        prev_dict_locset = Base.dict_loc_hostset(prev_searchmap)

        prev_removed, removed_map = self.gen_prev_dict_wo_instances(prev_dict_locset, removed_in_curr_tags_locset)

        if self.MTN_HOSTNAMES_FIX == "True":
            # Try to fix mtn hostnames in shardmap if it differs from hostlist
            for service,location in prev_removed.iteritems():
                for location, inum in location.iteritems():
                    for setdata in inum.values():
                        for hst in setdata:
                            for check_hst in curr_tagset[location]:
                                if ((hst["hostname"] == check_hst["hostname"]) and (hst["port"] == check_hst["port"])) and (hst["mtn_backbone_hostname"] != check_hst["mtn_backbone_hostname"]):
                                    log.warning("MTN HOSTNAME MISMATCH. Oldmap info: {0} {1} {2} Newmap info: {3} {4} {5}".format(
                                        hst["hostname"],
                                        hst["port"],
                                        hst["mtn_backbone_hostname"],
                                        check_hst["hostname"],
                                        check_hst["port"],
                                        check_hst["mtn_backbone_hostname"]))
                                    hst["mtn_backbone_hostname"] = check_hst["mtn_backbone_hostname"]

        current_searchmap_dict = self.gen_current_searchmap_dict(prev_removed,
                                                                 removed_map,
                                                                 add_in_curr_tags_locset,
                                                                 log,
                                                                 replicafactor,
                                                                 fat=fat)

        # copy current searchmap_dict with sets of readonly hashdict
        current_searchmap_dict_copy = {}

        for service, service_data in current_searchmap_dict.iteritems():
            if not current_searchmap_dict_copy.get(service, False):
                current_searchmap_dict_copy[service] = {}
            for gencfg_group, sharddata in service_data.iteritems():
                if not current_searchmap_dict_copy[service].get(gencfg_group, False):
                    current_searchmap_dict_copy[service][gencfg_group] = {}
                for shardnum, hostset in sharddata.iteritems():
                    if not current_searchmap_dict_copy[service][gencfg_group].get(shardnum, False):
                        current_searchmap_dict_copy[service][gencfg_group][shardnum] = set()
                    for hostdict in hostset:
                        temp_dict = copy.deepcopy(dict(hostdict))
                        temp_dict_hash = HashDict(temp_dict)
                        current_searchmap_dict_copy[service][gencfg_group][shardnum].add(temp_dict_hash)

        current_recluster_searchmap_dict = self.gen_current_recluster_searchmap_dict(current_searchmap_dict,
                                                                                     removed_map)

        return current_recluster_searchmap_dict, current_searchmap_dict_copy

    @staticmethod
    def dict_loc_hostset(hostlist):
        splitted_hosts = {}
        for el in hostlist:
            if not splitted_hosts.get(el['service'], False):
                splitted_hosts[el['service']] = {}
            if not splitted_hosts[el['service']].get(el['group'], False):
                splitted_hosts[el['service']][el['group']] = {}
            if not splitted_hosts[el['service']][el['group']].get(el['iNum'], False):
                splitted_hosts[el['service']][el['group']][el['iNum']] = set()

            tempdict = {key: str(el[key]) for key in Base.info_fields_hostlist}
            splitted_hosts[el['service']][el['group']][el['iNum']].add(HashDict(copy.deepcopy(tempdict),
                                                                                Base.hashing_map_fields))
        return splitted_hosts

    @staticmethod
    def make_location_set(hostlist):
        # Make dict with location as key and set() with Hashdict as value
        tag_locset = {}
        for el in hostlist:
            if not tag_locset.get(el['group'], False):
                tag_locset[el['group']] = set()
            tempdict = {key: str(el[key]) for key in Base.info_fields_hostlist}
            tag_locset[el['group']].add(HashDict(copy.deepcopy(tempdict),
                                                 Base.hashing_map_fields))
        # return location:set(Hashdict1, Hashdict2)
        return tag_locset

    @staticmethod
    def set_diff(dict_set1, dict_set2):
        # return dict with differences where keys - is locations
        diff_location_set = {}
        for location, tags_set in dict_set1.iteritems():
            diff_location_set[location] = dict_set1[location] - dict_set2[location]
        return diff_location_set

    @staticmethod
    def gen_prev_dict_wo_instances(prev_dict, removed_in_curr_tags_locset):
        removed_map = {}
        for service, location_dict in prev_dict.iteritems():
            removed_map[service] = {}
            for location, inum_dict in location_dict.iteritems():
                removed_map[service][location] = {}
                for inum, hostset in inum_dict.iteritems():
                    to_remove = prev_dict[service][location][inum] & removed_in_curr_tags_locset[location]
                    prev_dict[service][location][inum] = prev_dict[service][location][inum] - removed_in_curr_tags_locset[location]
                    removed_map[service][location][inum] = to_remove
                    if len(removed_map[service][location][inum]) > 0:
                        pass
                        # self.log.info("Hosts was removed. "
                        #               "From  service: {0} "
                        #               "location: {1} "
                        #               "inum: {2} "
                        #               "Count: {3} "
                        #               "Hosts:{4}".format(service,
                        #                                  location,
                        #                                  inum,
                        #                                  len(removed_map[service][location][inum]),
                        #                                  removed_map[service][location][inum]))
        return prev_dict, removed_map

    def gen_current_searchmap_dict(self, prev_removed, removed_map, add_in_curr_tags_locset, log, replicafactor, fat=False):
        # we need the same list in any location, sort for idempotency
        add_in_curr_tags_loclist_srt = {}

        for location, hostset in add_in_curr_tags_locset.iteritems():
            add_in_curr_tags_loclist_srt[location] = sorted(list(add_in_curr_tags_locset[location].copy()))

        for service, location_dict in prev_removed.iteritems():
            for location, inum_dict in location_dict.iteritems():
                copy_location_set = add_in_curr_tags_loclist_srt[location][:]

                # add hosts to removed places - to the same inum in different services
                # (because we use sorted list of hosts)
                # inumlist for fat shards is predefined
                if fat:
                    inumlist = [self.INUMFAT]
                else:
                    inumlist = range(self.INUM_START, self.INUM_END)

                for inum_n in inumlist:
                    inum = str(inum_n)
                    try:
                        if len(removed_map[service][location][inum]) == 0:
                            pass
                        for removed_num in range(len(removed_map[service][location][inum])):
                            # Check and continue - if we have all needed instances into location,
                            # regarding location replica factor
                            if len(prev_removed[service][location][inum]) >= self.GROUPS[location]:
                                log.info("WARNING. Pass adding instances because replicafactor reached for location."
                                         "Service: {0}, location: {1}, inum: {2}".format(service, location, inum))
                                continue
                            # Here we pop from current location sorted list and add it to removed vlan
                            hostdata_add = copy_location_set.pop()
                            prev_removed[service][location][inum].add(hostdata_add)
                            # TODO: fix this dirty check to avoid this case
                            for service_check, location_dict_check in prev_removed.iteritems():
                                for location_check, inum_dict_check in location_dict_check.iteritems():
                                    if location_check == location:
                                        # pass current location, check only different
                                        continue
                                    for hostdata_check in prev_removed[service_check][location_check][inum]:
                                        if hostdata_check["hostname"] == hostdata_add['hostname']:
                                            raise MageAPIServerHardError(
                                                "iNum: {0} for different locations presented on one hostname: {1}."
                                                "Location: {2} Location error: {3}."
                                                "Generation error. We found on the same host two same inums.".format(
                                                    inum,
                                                    hostdata_add['hostname'],
                                                    location,
                                                    location_check))

                    except KeyError:
                        if len(prev_removed[service][location][inum]) == 0:
                            raise MageAPIServerHardError(
                                "After removing we have only 0 hosts for inum {0} in location {1}, ".format(
                                    inum,
                                    location))
                        else:
                            log.info("We remove more hosts than added, "
                                     "but its okay because we have {0} "
                                     "replicas and replicafactor is {1}".format(
                                        len(prev_removed[service][location][inum]),
                                        replicafactor))

                            # #add free hosts
                            # for inum, hostset in inum_dict.iteritems():
                            #    try:
                            #        hostdata_add = copy_location_set.pop()
                            #        if len(prev_removed[service][location][inum]) < replicafactor + 1:
                            #            prev_removed[service][location][inum].add(hostdata_add)
                            #    except KeyError:
                            #        # if list with free host are empty - pass error, all host were added
                            #        pass

                            # if len(copy_location_set) > 0:
                            #    raise MageAPIServerHardError(
                            #        "Sorry, but we found free hosts after searchmap gen."
                            #        "Something going wrong. Location: {0}")

        return prev_removed

    @staticmethod
    def gen_current_recluster_searchmap_dict(current_searchmap, removed_map):
        for service, location_dict in current_searchmap.iteritems():
            for location, inums in location_dict.iteritems():
                for inum, inumdata in inums.iteritems():
                    if not removed_map[service][location].get(inum, False):
                        pass
                    else:
                        for el in removed_map[service][location][inum]:
                            current_searchmap[service][location][inum].add(el)
        return current_searchmap

    def gen_fat_wholemap(self, searchmap_dict, oauth_token=False, from_scratch=False):
        current_result_searchmap_fat = []

        for service_desc in self.SERVICES_GENSETTINGS:
            # If we get service with parent param - get instances from parent service
            if from_scratch:
                # Create searchmap from same instances for all services
                service_searchmap = searchmap_dict
            # We need get base shard services from prev searchmap
            else:
                if service_desc.get('parent', False):
                    service_searchmap = self.get_byshard_instances(searchmap_dict[service_desc['parent']],
                                                                   self.FAT_SHARDLIST, maptype='fat')
                else:
                    service_searchmap = self.get_byshard_instances(searchmap_dict[service_desc['service']],
                                                                   self.FAT_SHARDLIST, maptype='fat')

            mail_split_uncovered = self.zoo_split_uncover(service_desc['queue_ranges'],
                                                          self.FAT_GROUPS,
                                                          oauth_token=oauth_token)


            for shard_id, instance_group in service_searchmap.iteritems():
                # for every fat baseshard we gen searchmap with specified offset
                data = Base.gen_for_shard(1,
                                          shard_id,
                                          instance_group,
                                          service_desc["indexer_port_offset"],
                                          service_desc["search_port_offset"],
                                          service_desc["queue_id_port_offset"],
                                          service_desc["dump_port_offset"],
                                          False,
                                          service_desc["service"],
                                          self.PROJECT,
                                          mail_split_uncovered,
                                          self.SHARD_COUNT,
                                          base_port=service_desc.get('base_port', False),
                                          mapType=self.FATMAPTYPE,
                                          inumFat=self.INUMFAT
                                          )

                current_result_searchmap_fat = current_result_searchmap_fat + data
        return current_result_searchmap_fat

    def gen_wholemap(self, searchmap_dict, project, oauth_token=False, from_scratch=False):
        current_result_searchmap = []

        for service_desc in self.SERVICES_GENSETTINGS:
            # If we get service with parent param - get instances from parent service
            if from_scratch:
                # Create searchmap from same instances for all services
                service_searchmap = searchmap_dict
            else:
                # Get instances list from parent or service groups and prepare for genmap
                # TODO: We can use the same service_searchmap for all services because inum doesn't changed?
                if service_desc.get('parent', False):
                    service_searchmap = self.get_byshard_instances(searchmap_dict[service_desc['parent']], range(self.INUM_START, self.INUM_END))
                else:
                    service_searchmap = self.get_byshard_instances(searchmap_dict[service_desc['service']], range(self.INUM_START, self.INUM_END))

            mail_split_uncovered = self.zoo_split_uncover(service_desc['queue_ranges'],
                                                          self.FAT_GROUPS,
                                                          oauth_token=oauth_token)

            current_result_searchmap += (self.genmap(service_desc.get('base_port',False),
                                                    int(service_desc['indexer_port_offset']),
                                                    int(service_desc['search_port_offset']),
                                                    int(service_desc['queue_id_port_offset']),
                                                    int(service_desc['dump_port_offset']),
                                                    int(service_desc['offset']),
                                                    service_desc.get('oldstyle', False),
                                                    service_desc['service'],
                                                    service_searchmap,
                                                    self.REPLICAFACTOR,
                                                    project,
                                                    mail_split_uncovered,
                                                    ))
            self.log.info("Searchmap generation for service {0} completed. Current sm length:{1}".format(service_desc['service'],
                                                                                                         len(current_result_searchmap)))

        return current_result_searchmap


    @staticmethod
    def generate_byshard_instances(servicedict, shardlist, groups, fat=False):
        """
        In opposite for get_byshard_instances we generate instances shardmap manually for every shard
        (basing on replica factor).
        We need to do that because we generate single searchmap from scratch.
        """

        shard_dict = {}

        # Add hosts to shard by replica factor
        if fat:
            # Fat shard generation - differs from original
            # For any location we MUST clone hosts set, and than - pop elements from it.
            # We do that because we want same hostlist for all fat shards .
            # Also we want check - is replicafactor correct?
            for shard_id in shardlist:
                shard_dict[shard_id] = set()
                for location, replica_f in groups.iteritems():
                    temp_curr_tagset = set(servicedict[location])
                    for replica_factor in range(groups[location]):
                        shard_dict[shard_id].add(temp_curr_tagset.pop())
                    if len(temp_curr_tagset) > 0:
                        raise MageAPIServerHardError(
                            "We have some free hosts (replicafactor smaller than fat hosts count). "
                            "Looks like a mistake: {0}".format(str(temp_curr_tagset)))
        else:
            # Typical shard generation
            for shard_id in shardlist:
                shard_dict[shard_id] = set()
                for location, replica_f in groups.iteritems():
                    for replica_factor in range(groups[location]):
                        shard_dict[shard_id].add(servicedict[location].pop())

        return shard_dict

    def get_byshard_instances(self, service_datadict, shard_id_list, maptype='thin'):
        """
        Remove location, add instances to shard inum.
        """

        shard_dict = {}

        # For fat inum - get all hosts by special inum (self.INUMFAT)
        # and write it to shard from shardIdList
        for shard_id in shard_id_list:
            if not shard_dict.get(shard_id, False):
                shard_dict[shard_id] = set()
            for location, data in service_datadict.iteritems():
                if maptype == 'fat':
                    inum = self.INUMFAT
                else:
                    inum = shard_id
                for el in data[str(inum)]:
                    shard_dict[shard_id].add(el)
        return shard_dict


    @staticmethod
    def getzkconfig(shard, mail_split_uncovered):
        queue_line = False
        for lucene_split_range, lucene_queue_line in mail_split_uncovered.iteritems():
            if queue_line:
                break

            if int(shard) in range(int(lucene_split_range[0]), int(lucene_split_range[1]) + 1):
                queue_line = lucene_queue_line
            else:
                pass

        if not queue_line:
            raise MageAPIServerHardError("Cannot find queueline for fat shard:{0}".format(shard))
        return queue_line

    @staticmethod
    def split_thinmap(thin_searchmap, fatshards):
        # function receives two lists - with searchmap hosts, and with fat shards numbers
        # and return two lists - list with hosts not contains fatshards (partialmap), and list with hosts contains fat shard (partialmap_fat)
        partialmap = []
        partialmap_fat = set()
        for hst in thin_searchmap:
            # u'shards': u'0-30'
            hst_shards = range(int(hst["shards"].split("-")[0]), int(hst["shards"].split("-")[1]) + 1)
            flag = False
            for fatshard in fatshards:
                if int(fatshard) in hst_shards:
                    # So, we know that our fat shard in this range
                    # We need split this line to consecutives ranges
                    # Now - copy to partialmap_fat set
                    tmp_hst = copy.deepcopy(hst)
                    tmp_hst["_id"] = ObjectId()
                    partialmap_fat.add(HashDict(tmp_hst))
                    flag = True
            if not flag:
                partialmap.append(hst)
        return partialmap, partialmap_fat

    @staticmethod
    def wo_fatshards_map(fullmap_fat, fatshards):
        # function cut fat shard from shards range for host in hostlist
        # and return list with hosts with all consecutives ranges for shard
        changed_map = []
        for hst in fullmap_fat:
            hst_shards = set(range(int(hst["shards"].split("-")[0]), int(hst["shards"].split("-")[1]) + 1))
            for fatshard in fatshards:
                hst_shards = hst_shards - set([int(fatshard)])

            shrd_groups = Base.group_consecutives(hst_shards)
            for shrd_group in shrd_groups:
                hst_temp = copy.deepcopy(dict(hst))
                hst_temp["_id"] = ObjectId()
                hst_temp["shards"] = "{0}-{1}".format(shrd_group[0], shrd_group[-1])
                changed_map.append(hst_temp)
        return changed_map
