#!/usr/bin/env python                                                                                                                                         
# -*- coding: utf-8 -*-
import sys

import tornado.httpserver as httpserver
import tornado.ioloop as ioloop
import tornado.web as web
import logging.handlers
import os
from yaml import load, SafeLoader
from optparse import OptionParser
import glob

from libs.context.generator import Generator
from handlers.generator.GenmapHandler import GenmapHandler
from handlers.generator.GetmapHandler import GetmapHandler
from handlers.generator.GenmapReclusterHandler import GenmapReclusterHandler
from handlers.generator.LastrevHandler import LastrevHandler
from handlers.generator.UploadHandler import UploadHandler
from handlers.generator.SyncHostsHandler import SyncHostsHandler
from handlers.generator.SyncmapHandler import SyncmapHandler

# common
from handlers.PingHandler import PingHandler

from libs.backends.MongoDb import MongoDb
import motor.motor_tornado

def load_config(config):
    try:
        with open(config) as f:
            conf = load("".join(f.readlines()),  Loader=SafeLoader)
    except EnvironmentError:
        app_log.info("Can't open/parse config file: {0}".format(config))
        sys.exit(1)

    return conf

def full_withEnv(stringvar):
    stringvar = stringvar % os.environ
    return stringvar


if __name__ == '__main__':
    parser = OptionParser()
    parser.add_option("-c", "--config", type="string", dest="config",
                      metavar="CONFIG", default="config_prod.yaml",
                      help="YAML config file (default: %default)")
    (options, args) = parser.parse_args()

    conf = load_config(options.config)

    connectUri = full_withEnv(conf['database']['connectUri'])
    mongoDatabase = full_withEnv(conf['database']['mongoDatabase'])
    name = full_withEnv(conf['base']['name'])

    httpServerPort = int(full_withEnv(conf['server']['httpServerPort']))
    serverThreads = int(full_withEnv(conf['server']['serverThreads']))
    oauth_token = str(full_withEnv(conf['settings']['oauth_token']))
    full_log_path = full_withEnv(conf['log']['logpath'])
    log_format = conf['log']['format']
    stdout_log = conf['log']['stdout']

    if stdout_log == "False":
        stdout_log = False

    logging.basicConfig(level=logging.DEBUG,
                        format=log_format)

    full_filehandler = logging.handlers.WatchedFileHandler(full_log_path)
    full_filehandler.setFormatter(logging.Formatter(log_format))

    # TODO: we need one global request identifier

    access_log = logging.getLogger("tornado.access")
    app_log = logging.getLogger("tornado.application")
    gen_log = logging.getLogger("tornado.general")

    access_log.propagate = stdout_log
    app_log.propagate = stdout_log
    gen_log.propagate = stdout_log

    access_log.addHandler(full_filehandler)
    app_log.addHandler(full_filehandler)
    gen_log.addHandler(full_filehandler)

    log = logging.getLogger("tornado.application")

    settings = {
        "static_path": os.path.join(os.path.dirname(__file__), conf['settings']['staticDir']),
        "cookie_secret": conf['settings']['cookieSecret'],
        "login_url": conf['settings']['loginUrl'],
        "xsrf_cookies": False,
        "log": app_log,
        "default_filename": conf['settings']['indexfilename'],
        "serve_traceback": True,
        "debug": conf['settings']['debug'],
        "oauth_token": oauth_token,
        "configdir": conf['settings']['generator_configs']
    }

    # Specify handlers

    default_services_handlers = [(r"/css/(.*)", web.StaticFileHandler, {"path": settings['static_path'] + "/css/"}),
                                 (r"/scripts/(.*)", web.StaticFileHandler,
                                  {"path": settings['static_path'] + "/scripts/"}),
                                 (r"/views/(.*)", web.StaticFileHandler, {"path": settings['static_path'] + "/views/"}),
                                 (r"/ping", PingHandler),
                                 (r"/()", web.StaticFileHandler, {"path": settings['static_path'],
                                                                  "default_filename": settings['default_filename']}),
                                 ]

    services_handlers = []

    services_handlers = []
    for config in glob.glob(settings["configdir"]):
        generator = Generator(load_config(config))

        services_handlers.append((r'/api/v3.0/{project}/genmap'.format(project=generator.PROJECT), GenmapHandler, {"generator": generator}))
        services_handlers.append((r'/api/v3.0/{project}/genmaprec'.format(project=generator.PROJECT), GenmapReclusterHandler, {"generator": generator}))
        services_handlers.append((r'/api/v3.0/{project}/getmap'.format(project=generator.PROJECT), GetmapHandler, {"generator": generator}))
        services_handlers.append((r'/api/v3.0/{project}/lastrev'.format(project=generator.PROJECT), LastrevHandler, {"generator": generator}))
        services_handlers.append((r'/api/v3.0/{project}/upload'.format(project=generator.PROJECT), UploadHandler, {"generator": generator}))

        # Legacy
        # services_handlers.append((r'/api/v1.0/{project}/genmap'.format(project=generator.PROJECT), GenmapHandler, {"generator": generator}))
        # services_handlers.append((r'/api/v1.0/{project}/genmaprec'.format(project=generator.PROJECT), GenmapReclusterHandler, {"generator": generator}))
        services_handlers.append((r'/api/v1.0/{project}/getmap'.format(project=generator.PROJECT), GetmapHandler, {"generator": generator}))
        services_handlers.append((r'/api/v1.0/{project}/lastrev'.format(project=generator.PROJECT), LastrevHandler, {"generator": generator}))

        # services_handlers.append((r'/api/v2.0/{project}/genmap'.format(project=generator.PROJECT), GenmapHandler, {"generator": generator}))
        # services_handlers.append((r'/api/v2.0/{project}/genmaprec'.format(project=generator.PROJECT), GenmapReclusterHandler, {"generator": generator}))
        services_handlers.append((r'/api/v2.0/{project}/getmap'.format(project=generator.PROJECT), GetmapHandler, {"generator": generator}))
        services_handlers.append((r'/api/v2.0/{project}/lastrev'.format(project=generator.PROJECT), LastrevHandler,{"generator": generator}))

        services_handlers.append((r'/api/v1.0/{project}/synchosts'.format(project=generator.PROJECT), SyncHostsHandler, {"generator": generator}))
        services_handlers.append((r'/api/v1.0/{project}/syncmap'.format(project=generator.PROJECT), SyncmapHandler, {"generator": generator}))
        #services_handlers.append((r'/api/v1.0/mail/upload', MailUploadHandler))
        #services_handlers.append((r'/api/v1.0/disk/upload', DiskUploadHandler))

    app = web.Application(default_services_handlers + services_handlers, **settings)

    # Initialize web server
    try:
        server = httpserver.HTTPServer(app)
        server.bind(httpServerPort)
        server.start(serverThreads)
        app.settings["db"] = MongoDb(motor.motor_tornado.MotorClient(connectUri, readPreference='nearest')[mongoDatabase])
        ioloop.IOLoop.current().start()
    except KeyboardInterrupt:
        app_log.info("Stopping process by cntrl+c")
        ioloop.IOLoop.instance().stop()
