import os

import time
import nanny_rpc_client
from infra.nanny.yp_lite_api.proto import pod_sets_api_pb2
from infra.nanny.yp_lite_api.py_stubs import pod_sets_api_stub

from optparse import OptionParser

SERVICE_ID = 'disk-search-backend-yp-prod'
# SERVICE_PODNAME_PATTERN = "disk-search-backend-prod"
PODSET_ACL = ["abc:service:998", "okkk"]
ROOT_VOLUME_CLASS = "hdd"
NETWORK_MACRO = "_DISKPROD_SEARCH_NETS_"
CPU = {
    "SAS": {"CPU_GUARANTEE": 4900,
            "CPU_LIMIT": 4900},
    "MAN": {"CPU_GUARANTEE": 4900,
            "CPU_LIMIT": 4900},
    "VLA": {"CPU_GUARANTEE": 4900,
            "CPU_LIMIT": 4900},
    "VLAEXT": {"CPU_GUARANTEE": 4901,
               "CPU_LIMIT": 4901},
}

# to avoid situation whith insufficient cpu on DOM0 due rounding errors
CPU_ROUND_ERR = {
    "SAS": {"CPU_GUARANTEE": 4900,
                "CPU_LIMIT": 4900},
    "MAN": {"CPU_GUARANTEE": 4900,
                "CPU_LIMIT": 4900},
    "VLA": {"CPU_GUARANTEE": 4900,
                "CPU_LIMIT": 4900},
    "VLAEXT": {"CPU_GUARANTEE": 4901,
               "CPU_LIMIT": 4901}
}

ROUND_ERR_AFFECTED_HOSTLIST = {"man3-6223.search.yandex.net", "man0-2512.search.yandex.net",
                               "man0-2513.search.yandex.net", "man0-2516.search.yandex.net",
                               "man0-2523.search.yandex.net", "man0-2529.search.yandex.net",
                               "man0-2552.search.yandex.net", "man0-2553.search.yandex.net",
                               "man0-2554.search.yandex.net", "man0-2555.search.yandex.net",
                               "man0-2592.search.yandex.net", "man0-2595.search.yandex.net",
                               "man0-2597.search.yandex.net", "man0-2601.search.yandex.net",
                               "man0-2604.search.yandex.net", "man0-2619.search.yandex.net",
                               "man0-2622.search.yandex.net", "man0-2626.search.yandex.net",
                               "man0-2637.search.yandex.net", "man0-2644.search.yandex.net",
                               "man0-2649.search.yandex.net", "man0-2650.search.yandex.net",
                               "man0-2651.search.yandex.net", "man0-2669.search.yandex.net",
                               "man0-2673.search.yandex.net", "man0-2680.search.yandex.net",
                               "man0-2701.search.yandex.net", "man0-2707.search.yandex.net",
                               "man0-2710.search.yandex.net", "man0-2716.search.yandex.net",
                               "man0-2732.search.yandex.net", "man0-2739.search.yandex.net",
                               "man0-2740.search.yandex.net", "man0-2742.search.yandex.net",
                               "man0-2767.search.yandex.net", "man0-2778.search.yandex.net",
                               "man0-2782.search.yandex.net", "man0-2786.search.yandex.net",
                               "man0-2794.search.yandex.net", "man0-2805.search.yandex.net",
                               "man0-2830.search.yandex.net", "man1-1116.search.yandex.net",
                               "man1-1233.search.yandex.net", "man1-1273.search.yandex.net",
                               "man1-3281.search.yandex.net", "man1-4214.search.yandex.net",
                               "man1-4269.search.yandex.net", "man1-4348.search.yandex.net",
                               "man1-4744.search.yandex.net", "man1-4748.search.yandex.net",
                               "man1-4853.search.yandex.net", "man1-5101.search.yandex.net",
                               "man1-5402.search.yandex.net", "man1-5594.search.yandex.net",
                               "man1-5704.search.yandex.net", "man1-5865.search.yandex.net",
                               "man1-6628.search.yandex.net", "man1-7475.search.yandex.net",
                               "vla1-7943.search.yandex.net", "vla1-7947.search.yandex.net",
                               "vla1-7957.search.yandex.net"
                               }

ANON_MEM_LIMIT = 19456
MEM_GUARANTEE = 20480
ROOT_BW_GUARANTEE = 4
ROOT_BW_LIMIT = 20
ROOT_FS_QUOTA_GB = 2
WORKDIR_FS_QUOTA_GB = 3
NETWORK_BW_LIMIT = 50
NETWORK_BW_GUARANTEE = 40
ABC_SERVICE_QUOTA = 998
SEGMENT_ID = {"SAS": "base-search-cohabitation",
              "MAN": "base-search-cohabitation",
              "VLA": "yt_arnold_colocation",
              "VLAEXT": "yt_arnold_colocation"
              }


SNAPSHOTS_COUNT = 5

volumes = [
    {
        "disk_quota_gigabytes": 50,
        "mount_point": "/cores",
        "storage_class": "hdd",
        "bw_guarantee": 1,
        "bw_limit": 20,
    },
    {
        "disk_quota_gigabytes": 50,
        "mount_point": "/logs",
        "storage_class": "hdd",
        "bw_guarantee": 8,
        "bw_limit": 20,
    },
    {
        "disk_quota_gigabytes": 850,
        "mount_point": "/ssd",
        "storage_class": "ssd",
        "bw_guarantee": 50,
        "bw_limit": 50,
    }
]


# Firstly we create podset if success - than we can create pods
def init(cluster):
    c = nanny_rpc_client.RetryingRpcClient(rpc_url='https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/pod-sets/',
                                           oauth_token=os.getenv('OAUTH'))
    stub = pod_sets_api_stub.YpLiteUIPodSetsServiceStub(c)

    create_pod_set_req = pod_sets_api_pb2.CreatePodSetRequest()

    create_pod_set_req.service_id = SERVICE_ID

    create_pod_set_req.acl.type = 1
    acl_req = create_pod_set_req.acl.acl.add()
    acl_req.action = 1
    acl_req.permissions.append(2)
    for acl_str in PODSET_ACL:
        acl_req.subjects.append(acl_str)

    create_pod_set_req.allocation_request.snapshots_count = SNAPSHOTS_COUNT
    create_pod_set_req.allocation_request.root_volume_storage_class = ROOT_VOLUME_CLASS
    create_pod_set_req.allocation_request.replicas = 1
    create_pod_set_req.allocation_request.root_fs_quota_gigabytes = ROOT_FS_QUOTA_GB
    create_pod_set_req.allocation_request.work_dir_quota_gigabytes = WORKDIR_FS_QUOTA_GB
    create_pod_set_req.allocation_request.network_macro = NETWORK_MACRO
    create_pod_set_req.allocation_request.vcpu_guarantee = CPU[cluster]["CPU_GUARANTEE"]
    create_pod_set_req.allocation_request.vcpu_limit = CPU[cluster]["CPU_LIMIT"]
    create_pod_set_req.allocation_request.memory_guarantee_megabytes = MEM_GUARANTEE
    create_pod_set_req.allocation_request.anonymous_memory_limit_megabytes = ANON_MEM_LIMIT
    create_pod_set_req.allocation_request.pod_naming_mode = 1
    create_pod_set_req.allocation_request.root_bandwidth_guarantee_megabytes_per_sec = ROOT_BW_GUARANTEE
    create_pod_set_req.allocation_request.root_bandwidth_limit_megabytes_per_sec = ROOT_BW_LIMIT

    for volume in volumes:
        v_req = create_pod_set_req.allocation_request.persistent_volumes.add()
        v_req.disk_quota_gigabytes = volume["disk_quota_gigabytes"]
        v_req.mount_point = volume["mount_point"]
        v_req.storage_class = volume["storage_class"]
        v_req.bandwidth_guarantee_megabytes_per_sec = volume["bw_guarantee"]
        v_req.bandwidth_limit_megabytes_per_sec = volume["bw_limit"]

    # just for pod creation
    create_pod_set_req.allocation_request.labels.add(key="shard_id", value="0")

    create_pod_set_req.antiaffinity_constraints.node_max_pods = 6
    create_pod_set_req.antiaffinity_constraints.rack_max_pods = 20
    create_pod_set_req.antiaffinity_constraints.pod_group_id_path = "/labels/shard_id"

    create_pod_set_req.cluster = cluster
    create_pod_set_req.quota_settings.abc_service_id = ABC_SERVICE_QUOTA
    create_pod_set_req.allocation_mode = 0
    create_pod_set_req.node_segment_id = SEGMENT_ID[cluster]
    # create_pod_set_req.node_affinity.network_capacity = 0
    # create_pod_set_req.node_affinity.require_avx = 0
    # create_pod_set_req.node_affinity.require_avx2 = 0
    # create_pod_set_req.node_affinity.sox_compliant = 0

    print(stub.create_pod_set(create_pod_set_req))

    remove_pod_req = pod_sets_api_pb2.RemovePodRequest()

    remove_pod_req.pod_id = f"{SERVICE_ID}-1"
    remove_pod_req.cluster = cluster

    print(stub.remove_pod(remove_pod_req))


def add(dom0hint: str, label: str, cluster: str, podnumber: str, spec: str):
    res_cluster = f"{cluster}{spec}"
    c = nanny_rpc_client.RetryingRpcClient(rpc_url='https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/pod-sets/',
                                           oauth_token=os.getenv('OAUTH'))
    stub = pod_sets_api_stub.YpLiteUIPodSetsServiceStub(c)
    req = pod_sets_api_pb2.CreateSpecificPodRequest()
    req.service_id = SERVICE_ID

    req.cluster = cluster

    req.allocation_request.allocation.snapshots_count = SNAPSHOTS_COUNT
    req.allocation_request.allocation.root_volume_storage_class = ROOT_VOLUME_CLASS
    req.allocation_request.allocation.replicas = 1
    req.allocation_request.allocation.root_fs_quota_gigabytes = ROOT_FS_QUOTA_GB
    req.allocation_request.allocation.work_dir_quota_gigabytes = WORKDIR_FS_QUOTA_GB
    req.allocation_request.allocation.network_macro = NETWORK_MACRO
    if dom0hint in ROUND_ERR_AFFECTED_HOSTLIST:
        print(f"USE OVERRIDED CPU SETTINGS FOR {dom0hint} to avoind rounding errors")
        req.allocation_request.allocation.vcpu_guarantee = CPU_ROUND_ERR[res_cluster]["CPU_GUARANTEE"]
        req.allocation_request.allocation.vcpu_limit = CPU_ROUND_ERR[res_cluster]["CPU_LIMIT"]
    else:
        req.allocation_request.allocation.vcpu_guarantee = CPU[res_cluster]["CPU_GUARANTEE"]
        req.allocation_request.allocation.vcpu_limit = CPU[res_cluster]["CPU_LIMIT"]
    req.allocation_request.allocation.memory_guarantee_megabytes = MEM_GUARANTEE
    req.allocation_request.allocation.anonymous_memory_limit_megabytes = ANON_MEM_LIMIT
    req.allocation_request.allocation.pod_naming_mode = 1
    req.allocation_request.allocation.root_bandwidth_guarantee_megabytes_per_sec = ROOT_BW_GUARANTEE
    req.allocation_request.allocation.root_bandwidth_limit_megabytes_per_sec = ROOT_BW_LIMIT
    req.allocation_request.allocation.network_bandwidth_limit_megabytes_per_sec = NETWORK_BW_LIMIT
    req.allocation_request.allocation.network_bandwidth_guarantee_megabytes_per_sec = NETWORK_BW_GUARANTEE

    podnum = int(podnumber)
    m_pod_id = f"{SERVICE_ID}-{podnum}"
    req.allocation_request.pod_specific_settings.pod_id = f"{m_pod_id}"
    hints = req.allocation_request.pod_specific_settings.scheduling.hints.add()
    hints.node_id = dom0hint
    hints.strong = True

    for volume in volumes:
        v_req = req.allocation_request.allocation.persistent_volumes.add()
        v_req.disk_quota_gigabytes = volume["disk_quota_gigabytes"]
        v_req.mount_point = volume["mount_point"]
        v_req.storage_class = volume["storage_class"]
        v_req.bandwidth_guarantee_megabytes_per_sec = volume["bw_guarantee"]
        v_req.bandwidth_limit_megabytes_per_sec = volume["bw_limit"]

    req.allocation_request.allocation.labels.add(key="shard_id", value=label)

    tries = 5
    for i in range(tries):
        try:
            answer = stub.create_specific_pod(req)
            print(f"ADDED: {m_pod_id} {dom0hint} {label} {answer}")
        except Exception as e:
            print(f"FAILED: {m_pod_id} {dom0hint} {label} {e}")
            if i < tries - 1:
                print(f"RETYING try {i}: {m_pod_id} {dom0hint} {label}")
                time.sleep(1)
                continue
            else:
                # print(f"TRIES NUMBER REACHED. Exiting. try {i}: {dom0hint} {label}")
                break
        break


def remove(cluster: str, podnumber: str):
    c = nanny_rpc_client.RetryingRpcClient(rpc_url='https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/pod-sets/',
                                           oauth_token=os.getenv('OAUTH'))
    stub = pod_sets_api_stub.YpLiteUIPodSetsServiceStub(c)
    req = pod_sets_api_pb2.RemovePodRequest()

    podnum = int(podnumber)
    m_pod_id = f"{SERVICE_ID}-{podnum}"
    req.pod_id = f"{m_pod_id}"
    req.cluster = cluster

    tries = 5
    for i in range(tries):
        try:
            answer = stub.remove_pod(req)
            print(f"REMOVED: {m_pod_id} {answer}")
        except Exception as e:
            print(f"FAILED: {m_pod_id} {e}")
            if i < tries - 1:
                print(f"RETYING try {i}: {m_pod_id}")
                time.sleep(1)
                continue
            else:
                # print(f"TRIES NUMBER REACHED. Exiting. try {i}: {dom0hint} {label}")
                break
        break


def set_state(pod: str, snapshot: str, state: str, cluster: str):
    c = nanny_rpc_client.RetryingRpcClient(rpc_url='https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/pod-sets/',
                                           oauth_token=os.getenv('OAUTH'))
    stub = pod_sets_api_stub.YpLiteUIPodSetsServiceStub(c)
    req = pod_sets_api_pb2.GetPodRequest()
    req.cluster = cluster
    req.pod_id = pod

    answer = stub.get_pod(req)

    nanny_version = ""
    for el in answer.pod.labels.attributes:
        if el.key == 'nanny_version':
            nanny_version = el.value.decode("utf-8")
            break
    print(f"DETECTED NANNY VERSION IS: {nanny_version}")

    c = nanny_rpc_client.RetryingRpcClient(rpc_url='https://yp-lite-ui.nanny.yandex-team.ru/api/yplite/pod-sets/',
                                           oauth_token=os.getenv('OAUTH'))
    stub = pod_sets_api_stub.YpLiteUIPodSetsServiceStub(c)
    req = pod_sets_api_pb2.SetInstanceTargetStateRequest()

    req.service_id = SERVICE_ID
    req.cluster = cluster
    req.pod_id = pod
    req.snapshot_id = snapshot
    req.target_state = state
    req.version = nanny_version
    req.snapshot_id = snapshot

    answer = stub.set_instance_target_state(req)

    print(f"TARGET STATE WAS CHANGED TO: {pod} {state} {snapshot} {answer}")


def main():
    parser = OptionParser()

    parser.add_option("-a", "--action",
                      type="string",
                      dest="action",
                      metavar="ACTION",
                      help="Specify one action: init, add")

    parser.add_option("--dom0",
                      type="string",
                      dest="dom0",
                      metavar="DOM0",
                      help="Create pod with specified dom0 as hint. Hint must be valid dom0 name from GENCFG group")

    parser.add_option("--label",
                      type="string",
                      dest="label",
                      metavar="label",
                      help="Create pod with specified label (inum). Label must be valid integer number.")

    parser.add_option("-p", "--pod",
                      type="string",
                      dest="pod",
                      metavar="pod",
                      help="Set state for specific pod.")

    parser.add_option("--snapshot",
                      type="string",
                      dest="snapshot",
                      metavar="snapshot",
                      help="Set state for specific snapshot.")

    parser.add_option("--state",
                      type="string",
                      dest="state",
                      metavar="state",
                      help="Set state.")

    parser.add_option("--cluster",
                      type="string",
                      dest="cluster",
                      metavar="cluster",
                      help="Set cluster (SAS,MAN,VLA).")

    parser.add_option("--spec",
                      type="string",
                      dest="spec",
                      metavar="spec",
                      help="Specify additional cluster spec (EXT).")

    parser.add_option("--podnumber",
                      type="string",
                      dest="podnumber",
                      metavar="podnumber",
                      help="Set pod number. Pod name will be: mail-search-backend-prod-{podnumber}")

    (options, args) = parser.parse_args()

    action = options.action
    dom0 = options.dom0
    label = options.label
    pod = options.pod
    snapshot = options.snapshot
    state = options.state
    cluster = options.cluster
    podnumber = options.podnumber
    spec = options.spec

    if not action:
        print("Error! Action required!")
    elif action == "init":
        if cluster:
            init(cluster)
        else:
            print("Error! specify cluster")
    elif action == "add":
        if dom0 and label and cluster and podnumber:
            # print(f"Add dom0 {dom0} label {label}")
            add(dom0, label, cluster, podnumber, spec)
        else:
            print("Error! For add action --dom0 and --label and --cluster and --podnumber option is required!")
    elif action == "remove":
        if cluster and podnumber:
            # print(f"Add dom0 {dom0} label {label}")
            remove(cluster, podnumber)
        else:
            print("Error! For remove action --cluster and --podnumber options is required!")
    elif action == "set_state":
        if pod and snapshot and state and cluster:
            set_state(pod, snapshot, state, cluster)
        else:
            print("Error! Please specify -p (pod) -snapshot (snapshot) -state (state) --cluster")


if __name__ == '__main__':
    main()
