package ru.yandex.tools.benchmark;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.Callable;

public class BenchmarkIteration {
    private final BenchmarkOptions options;
    private final int repeats;
    private final Object[] results;
    private long bestTiming;
    private Callable<?> bestBench;
    private int iterationNumber = 0;

    public BenchmarkIteration(
        final BenchmarkOptions options,
        final long bestTiming,
        final Callable<?> bestBench)
    {
        this.options = options;
        this.bestTiming = bestTiming;
        this.bestBench = bestBench;
        repeats = options.repeats();
        results = new Object[repeats];
    }

    public long bestTiming() {
        return bestTiming;
    }

    public Callable<?> bestBench() {
        return bestBench;
    }

    public long[] benchmark(
        final int cycleNumber,
        final List<? extends Callable<?>> benchs,
        final Object etalon)
        throws Exception
    {
        long[] timings = new long[benchs.size()];
        Arrays.fill(timings, Long.MAX_VALUE);
        ++iterationNumber;
        for (int j = 0; j < benchs.size(); ++j) {
            int index = (j + cycleNumber) % benchs.size();
            Callable<?> bench = benchs.get(index);
            long start = System.nanoTime();
            for (int k = repeats; --k >= 0;) {
                results[k] = bench.call();
            }
            long timing = System.nanoTime() - start;
            for (int k = repeats; --k >= 0;) {
                if (!Objects.equals(etalon, results[k])) {
                    throw new Exception(
                        "When performing iteration #" + iterationNumber
                        + " during cycle # " + cycleNumber
                        + " results mismatch occured. Expected: '"
                        + etalon + "' but was '" + results[k] + '\'');
                }
            }
            if (timing < timings[index]) {
                timings[index] = timing;
            }
            if (timing < bestTiming) {
                bestTiming = timing;
                bestBench = bench;
            }
        }
        if (options.shouldPrint()) {
            options.out().println(
                "At iteration #" + iterationNumber
                + " of benchmarking cycle #" + cycleNumber
                + " best timing is "
                + String.format(
                    "%.3f",
                    ((double) bestTiming) / repeats)
                + " ns per iteration held by " + bestBench);
        }
        return timings;
    }
}

