package ru.yandex.tools.benchmark.sets;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.Callable;
import java.util.function.Supplier;

import ru.yandex.tools.benchmark.Benchmark;
import ru.yandex.tools.benchmark.BenchmarkOptions;

public final class SetsBenchmark {
    private static final String[] STRINGS = new String[] {
        "http.connection",
        "http.request",
        "http.response",
        "http.target_host",
        "http.request_sent",
        "http.route",
        "http.protocol.redirect-locations",
        "http.cookiespec-registry",
        "http.cookie-spec",
        "http.cookie-origin",
        "http.cookie-store",
        "http.auth.credentials-provider",
        "http.auth.auth-cache",
        "http.auth.target-scope",
        "http.auth.proxy-scope",
        "http.user-token",
        "http.authscheme-registry",
        "http.request-config",
        "ru.yandex.http.util.nio.client.RequestsListener"
    };

    private static final Supplier<Callable<Integer>> TREE =
        new Supplier<Callable<Integer>>() {
            @Override
            public Callable<Integer> get() {
                return new TreeSetBenchmark();
            }
        };

    private static final Supplier<Callable<Integer>> LINKED =
        new Supplier<Callable<Integer>>() {
            @Override
            public Callable<Integer> get() {
                return new LinkedHashSetBenchmark();
            }
        };

    private static final Supplier<Callable<Integer>> HASH =
        new Supplier<Callable<Integer>>() {
            @Override
            public Callable<Integer> get() {
                return new HashSetBenchmark();
            }
        };

    private static final Supplier<Callable<Integer>> HASH2 =
        new Supplier<Callable<Integer>>() {
            @Override
            public Callable<Integer> get() {
                return new HashSet2Benchmark();
            }
        };

    private static final Supplier<Callable<Integer>> HASH4 =
        new Supplier<Callable<Integer>>() {
            @Override
            public Callable<Integer> get() {
                return new HashSet4Benchmark();
            }
        };

    private SetsBenchmark() {
    }

    public static void main(final String... args) throws Exception {
        final int defaultRepeats = 1000;
        final int defaultIterations = 10000;
        final int defaultWarmupIterations = 1000000;
        int repeats;
        if (args.length > 0) {
            repeats = Integer.parseInt(args[0]);
        } else {
            repeats = defaultRepeats;
        }
        int iterations;
        if (args.length > 1) {
            iterations = Integer.parseInt(args[1]);
        } else {
            iterations = defaultIterations;
        }
        int warmupIterations;
        if (args.length > 2) {
            warmupIterations = Integer.parseInt(args[2]);
        } else {
            warmupIterations = defaultWarmupIterations;
        }
        BenchmarkOptions options = new BenchmarkOptions()
            .repeats(repeats)
            .benchmarkIterations(iterations)
            .warmupIterations(warmupIterations);
        List<long[]> timings = new ArrayList<>();
        doBenchmark(options, timings);
        doBenchmark(options.gcSleep(0L), timings);
        System.out.println();
        for (long[] timing: timings) {
            for (long time: timing) {
                System.out.print(time);
                System.out.print(';');
            }
            System.out.println();
        }
    }

    private static void doBenchmark(
        final BenchmarkOptions options,
        final List<long[]> timings)
        throws Exception
    {
        System.out.println("\n\nOverall test:\n");
        timings.add(
            new Benchmark(
                options,
                Arrays.asList(HASH, HASH2, HASH4, LINKED, TREE))
                .benchmark());
        System.out.println("\n\nHashSet based test:\n");
        timings.add(
            new Benchmark(options, Arrays.asList(HASH, HASH2, HASH4, LINKED))
                .benchmark());
        System.out.println("\n\nPure HashSet test:\n");
        timings.add(
            new Benchmark(options, Arrays.asList(HASH, HASH2, HASH4))
                .benchmark());
    }

    private static class BaseBenchmark implements Callable<Integer> {
        private final Set<String> set;

        BaseBenchmark(final Set<String> set) {
            this.set = set;
        }

        @Override
        public Integer call() {
            int matches = 0;
            for (int i = 0; i < STRINGS.length; ++i) {
                if (set.contains(STRINGS[i])) {
                    ++matches;
                }
            }
            return matches;
        }
    }

    private static class TreeSetBenchmark extends BaseBenchmark {
        TreeSetBenchmark() {
            super(new TreeSet<>(Arrays.asList(STRINGS)));
        }

        @Override
        public String toString() {
            return "TreeSet";
        }
    }

    private static class LinkedHashSetBenchmark extends BaseBenchmark {
        LinkedHashSetBenchmark() {
            super(new LinkedHashSet<>(Arrays.asList(STRINGS)));
        }

        @Override
        public String toString() {
            return "LinkedHashSet";
        }
    }

    private static class HashSetBenchmark extends BaseBenchmark {
        HashSetBenchmark() {
            super(new HashSet<>(Arrays.asList(STRINGS)));
        }

        @Override
        public String toString() {
            return "HashSet";
        }
    }

    private static class HashSet2Benchmark extends BaseBenchmark {
        HashSet2Benchmark() {
            super(prepareSet());
        }

        private static Set<String> prepareSet() {
            Set<String> set = new HashSet<>(STRINGS.length << 1);
            set.addAll(Arrays.asList(STRINGS));
            return set;
        }

        @Override
        public String toString() {
            return "HashSet2";
        }
    }

    private static class HashSet4Benchmark extends BaseBenchmark {
        HashSet4Benchmark() {
            super(prepareSet());
        }

        private static Set<String> prepareSet() {
            Set<String> set = new HashSet<>(STRINGS.length << 2);
            set.addAll(Arrays.asList(STRINGS));
            return set;
        }

        @Override
        public String toString() {
            return "HashSet4";
        }
    }
}

