package ru.yandex.config.generator;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.github.javaparser.JavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.ImportDeclaration;
import com.github.javaparser.ast.Modifier;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.ast.body.Parameter;
import com.github.javaparser.ast.expr.AssignExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.FieldAccessExpr;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.expr.SimpleName;
import com.github.javaparser.ast.expr.StringLiteralExpr;
import com.github.javaparser.ast.expr.ThisExpr;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExplicitConstructorInvocationStmt;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.stmt.ReturnStmt;
import com.github.javaparser.ast.stmt.Statement;
import com.github.javaparser.ast.type.ClassOrInterfaceType;
import com.github.javaparser.ast.type.TypeParameter;

public class AbstractBuilderConfigGenerator
    extends AbstractConfigGenerator
{
    private static final Set<String> RESERVED_METHODS =
        new LinkedHashSet<>(Arrays.asList("build", "self"));
    private static final ClassOrInterfaceType T_TYPE =
        new ClassOrInterfaceType(null, "T");
    
    private final ClassOrInterfaceDeclaration builderClass;
    private final CompilationUnit unit;
    private final File builderFile;

    public AbstractBuilderConfigGenerator(
        final Config config)
        throws IOException
    {
        super(config, config.abstractBuilderName());
        this.builderFile =
            new File(config.directory(), configName + ".java");
        if (builderFile.exists()) {
            unit = JavaParser.parse(builderFile);
            Optional<ClassOrInterfaceDeclaration> optional =
                unit.getClassByName(configName);
            if (!optional.isPresent()) {
                throw new IOException(builderFile.getAbsolutePath()
                    + " exists but do not contain class inside "
                    + configName);
            }

            builderClass = optional.get();
        } else {
            unit = new CompilationUnit(
                config.getPackage(),
                new NodeList<>(config.unit().getImports()),
                new NodeList<>(),
                null);
            builderClass = unit.addClass(configName);
            builderClass.setAbstract(true);
            builderClass.addTypeParameter(
                new TypeParameter(
                    "T",
                    NodeList.nodeList(
                        new ClassOrInterfaceType(
                            null,
                            new SimpleName(configName),
                            NodeList.nodeList(T_TYPE)))));

            if (config.extendsName() != null) {
                String extendsClass =
                    "Abstract" + config.extendsName() + "Builder";
                builderClass.addExtendedType(extendsClass +"<T>");
                unit.addImport(
                    config.extendsPackage() + '.' + extendsClass);
            }
            builderClass.addImplementedType(config.name());
        }
    }

    @Override
    protected String fieldConfigString(final ConfigField field) {
        return field.builderType().asString();
    }

    protected void handleImports() {
        Collection<ImportDeclaration> imports = unit.getImports();
        imports.add(new ImportDeclaration(
            "ru.yandex.parser.config.IniConfig",
            false,
            false));
        imports.add(new ImportDeclaration(
            "ru.yandex.parser.config.ConfigException",
            false,
            false));
        imports.add(
            new ImportDeclaration(
                "ru.yandex.parser.config.ConfigException",
                false,
                false));

        handleConfigsImports(config, imports, false);
        handleCollectionsImports(imports);
        if (config.extendsName() != null) {
            handleImport(config.extendsName(), null, true, imports);
        }

        unit.setImports(new NodeList<>(imports));
        adjustImports(unit);
    }

    protected void handleCollectionsImports(
        final Collection<ImportDeclaration> imports)
    {
        for (ConfigField field: config.fields()) {
            switch (field.fieldType()) {
                case MAP:
                    imports.add(MAP_IMPORT);
                    break;
                case SET:
                    imports.add(SET_IMPORT);
                    break;
                case LIST:
                    imports.add(LIST_IMPORT);
                    break;
            }
        }
    }

    protected void handleClassFields() {
        for (ConfigField field: config.fields()) {
            Optional<FieldDeclaration> optional =
                builderClass.getFieldByName(field.name());
            if (!optional.isPresent()) {
                builderClass.addField(
                    field.builderType(),
                    field.name(),
                    Modifier.PRIVATE);
            }
        }
    }

    protected void handleFromConfigConstructor() {
        ConstructorDeclaration fromConfigContructor = null;

        for (ConstructorDeclaration md: builderClass.getConstructors()) {
            if (md.getParameters().size() == 1) {
                String first = md.getParameter(0).getType().asString();
                if (config.name().equalsIgnoreCase(first)) {
                    fromConfigContructor = md.asConstructorDeclaration();
                    break;
                }
            }
        }

        if (fromConfigContructor == null) {
            fromConfigContructor =
                builderClass.addConstructor(Modifier.PUBLIC);
            fromConfigContructor.addParameter(
                new Parameter(
                    FINAL_MODIFIER,
                    new ClassOrInterfaceType(
                        null,
                        config.name()),
                    new SimpleName("config")));

            NodeList<Statement> constrStmts = new NodeList<>();
            constrStmts.add(
                new ExplicitConstructorInvocationStmt(
                    false,
                    null,
                    NodeList.nodeList(new NameExpr("config"))));

            for (ConfigField field: config.fields()) {
                constrStmts.add(
                    new ExpressionStmt(
                        new MethodCallExpr(
                            field.name(),
                            new MethodCallExpr(
                                new NameExpr("config"),
                                field.name()))));
            }

            fromConfigContructor.setBody(new BlockStmt(constrStmts));
        }
    }

    protected void handleFromIniConstructor() {
        ConstructorDeclaration fromIniConstructor = null;

        for (ConstructorDeclaration md: builderClass.getConstructors()) {
            if (md.getParameters().size() == 2) {
                String first = md.getParameter(0).getType().asString();
                String second = md.getParameter(1).getType().asString();
                if ("IniConfig".equalsIgnoreCase(first)
                    && config.name().equalsIgnoreCase(second))
                {
                    fromIniConstructor = md.asConstructorDeclaration();
                    break;
                }
            }
        }

        NodeList<Statement> constrStmts = new NodeList<>();

        Set<String> initedFields = new LinkedHashSet<>();
        if (fromIniConstructor == null) {
            fromIniConstructor =
                builderClass.addConstructor(Modifier.PUBLIC);

            fromIniConstructor.addParameter(
                new Parameter(
                    FINAL_MODIFIER,
                    new ClassOrInterfaceType(
                        null,
                        "IniConfig"),
                    new SimpleName("config")));
            fromIniConstructor.addParameter(
                new Parameter(
                    FINAL_MODIFIER,
                    new ClassOrInterfaceType(
                        null,
                        config.name()),
                    new SimpleName("defaults")));

            fromIniConstructor.addThrownException(
                new ClassOrInterfaceType(
                    null,
                    "ConfigException"));

            if (config.extendsName() != null) {
                constrStmts.add(
                    new ExplicitConstructorInvocationStmt(
                        false,
                        null,
                        NodeList.nodeList(
                            new NameExpr("config"),
                            new NameExpr("defaults"))));
            }
        } else {
            constrStmts =
                fromIniConstructor.getBody().asBlockStmt().getStatements();

            for (Statement statement: constrStmts) {
                if (statement.isExpressionStmt()) {
                    Expression expression =
                        statement.asExpressionStmt().getExpression();
                    if (expression.isAssignExpr()) {
                        Expression target = expression.asAssignExpr().getTarget();
                        if (target.isFieldAccessExpr()) {
                            initedFields.add(
                                target.asFieldAccessExpr().getNameAsString());
                        }
                    }
                }
            }
        }

        for (ConfigField field: config.fields()) {
            if (!initedFields.contains(field.name())) {
                constrStmts.addAll(fieldIniConfigFetch(field));
            }
        }

        fromIniConstructor.setBody(new BlockStmt(constrStmts));
    }

    protected void handleMethods() {
        Map<String, ConfigField> fields = config.fields().stream().collect(
            Collectors.toMap(ConfigField::name, Function.identity()));

        Iterator<MethodDeclaration> methodIterator =
            builderClass.getMethods().iterator();

        Set<String> fieldsWithSetMethods = new LinkedHashSet<>();
        Set<String> fieldsWithGetMethods = new LinkedHashSet<>();

        while (methodIterator.hasNext()) {
            MethodDeclaration md = methodIterator.next();

            if (md.getModifiers().contains(Modifier.PRIVATE)) {
                continue;
            }

            if (RESERVED_METHODS.contains(md.getNameAsString())) {
                continue;
            }

            String typeName = md.getTypeAsString();
            if (!fields.containsKey(md.getNameAsString())) {
                if (md.getParameters().size() == 0
                    && !md.getType().isVoidType())
                {
                    if (askUser(
                        "Y/Yes remove method " + md.getNameAsString() + " from "
                            + configName))
                    {
                        builderClass.remove(md);
                    }
                } else if (md.getParameters().size() == 1
                    && typeName.length() == 1)
                {
                    if (askUser(
                        "Y/Yes remove method " + md.getNameAsString() + " from "
                            + configName))
                    {
                        builderClass.remove(md);
                    }
                }

                continue;
            }

            if (typeName.length() == 1
                || configName.equalsIgnoreCase(typeName))
            {
                fieldsWithSetMethods.add(md.getNameAsString());
            } else if (md.getParameters().size() == 0) {
                fieldsWithGetMethods.add(md.getNameAsString());
            } else {
                if (askUser("Delete method " + md.getNameAsString()
                    + " with return type " + md.getType().asString()
                    + " and params " + md.getParameters().toString()))
                {
                    methodIterator.remove();
                }
            }
        }

        for (ConfigField field: config.fields()) {
            if (!fieldsWithGetMethods.contains(field.name())) {
                MethodDeclaration methodDeclaration =
                    builderClass.addMethod(field.name(), Modifier.PUBLIC);
                methodDeclaration.setType(field.builderType());
                methodDeclaration.addAnnotation("Override");
                methodDeclaration.setBody(
                    new BlockStmt().addStatement(
                        new ReturnStmt(
                            new NameExpr(field.name()))));
            }

            if (!fieldsWithSetMethods.contains(field.name())) {
                MethodDeclaration methodDeclaration =
                    builderClass.addMethod(field.name(), Modifier.PUBLIC);
                methodDeclaration.setType(
                    new ClassOrInterfaceType(null, "T"));

                Parameter parameter = new Parameter(
                    FINAL_MODIFIER,
                    field.returnType(),
                    new SimpleName("value"));
                methodDeclaration.addParameter(parameter);
                methodDeclaration.setBody(setMethodBody(field));
            }
        }
    }

    @Override
    public CompilationUnit get() {
        handleImports();
        handleClassFields();

        handleFromConfigConstructor();
        handleFromIniConstructor();
        handleMethods();

        adjustImports(unit);
        adjustNodes(builderClass);
        return unit;
    }

    private String builderClass(final ConfigField field) {
        if (field.fieldType() == FieldType.CONFIG) {
            return field.returnType().asClassOrInterfaceType()
                .getName().asString() + "Builder";
        }

        return null;
    }

    private String fieldConfigName(final ConfigField field) {
        String s = field.name();
        if (field.fieldType() == FieldType.CONFIG && s.endsWith("Config")) {
            s = s.replaceAll("Config", "");
        }
        StringBuilder sb = new StringBuilder();
        for (int i = 0; i < s.length(); i++) {
            char c = s.charAt(i);
            if (Character.isLetter(c) && Character.isUpperCase(c)) {
                sb.append('-');
                sb.append(Character.toLowerCase(c));
            } else {
                sb.append(c);
            }
        }

        return sb.toString();
    }

    private Statement getConfigStatement(
        final ConfigField field,
        final String method,
        final Expression parser)
    {
        NodeList<Expression> args = new NodeList<>();
        args.add(new StringLiteralExpr(fieldConfigName(field)));
        args.add(
            new MethodCallExpr(
                new NameExpr("defaults"),
                new SimpleName(field.name())));
        if (parser != null) {
            args.add(parser);
        }

        return new ExpressionStmt(
            new AssignExpr(
                new FieldAccessExpr(
                    new ThisExpr(),
                    field.name()),
                new MethodCallExpr(
                    new NameExpr("config"),
                    method,
                    args),
                AssignExpr.Operator.ASSIGN));
    }

    private Statement collectionIniConfigFetch(
        final ConfigField field,
        final String collection)
    {
        unit.addImport("ru.yandex.parser.string.CollectionParser");
        Expression parser = new ObjectCreationExpr(
            null,
            new ClassOrInterfaceType(
                null,
                new SimpleName("CollectionParser"),
                new NodeList<>()),
            new NodeList<>(
                new NameExpr("String::trim"),
                new NameExpr(collection + "::new")));

        return getConfigStatement(field, "get", parser);
    }

    private List<Statement> fieldIniConfigFetch(final ConfigField field) {
        List<Statement> statements = new ArrayList<>();
        switch (field.fieldType()) {
            case IMMUTABLE:
                String methodName = "get";
                Expression parser = null;
                if (field.returnType().isPrimitiveType()) {
                    methodName = "get" + ConfigField.capitalize(
                        field.returnType().asString());
                } else if (field.returnType().isClassOrInterfaceType()
                    && field.returnType().asClassOrInterfaceType().isBoxedType())
                {
                    methodName = "get" + field.returnType().asString();
                } else if (field.returnType().isClassOrInterfaceType()) {
                    String typeString = field.returnType().asString();
                    if ("String".equalsIgnoreCase(typeString))
                    {
                        methodName = "getString";
                    } else if ("Pattern".equalsIgnoreCase(typeString)) {
                        parser = new NameExpr("Pattern::compile");
                    } else if ("IniConfig".equalsIgnoreCase(typeString)) {
                        methodName = "sectionOrDefault";
                    }
                }

                statements.add(
                    getConfigStatement(field, methodName, parser));
                break;
            case LIST:
                statements.add(
                    collectionIniConfigFetch(field, "ArrayList"));
                break;
            case SET:
                statements.add(
                    collectionIniConfigFetch(field, "LinkedHashSet"));
                break;
            case MAP:
                statements.add(
                    collectionIniConfigFetch(field, "LinkedHashMap"));
                break;
            case CONFIG:
                NodeList<Expression> configBuilderArgs =
                    new NodeList<>(
                        new MethodCallExpr(
                            new NameExpr("config"),
                            new SimpleName("section"),
                            NodeList.nodeList(
                                new StringLiteralExpr(
                                    fieldConfigName(field)))),
                        new MethodCallExpr(
                            new NameExpr("defaults"),
                            new SimpleName(field.name())));
                statements.add(
                    new ExpressionStmt(
                        new AssignExpr(
                            new FieldAccessExpr(
                                new ThisExpr(),
                                field.name()),
                            new ObjectCreationExpr(
                                null,
                                new ClassOrInterfaceType(
                                    null,
                                    builderClass(field)),
                                configBuilderArgs),
                            AssignExpr.Operator.ASSIGN)));
        }

        return statements;
    }

    private BlockStmt setMethodBody(final ConfigField field) {
        switch (field.fieldType()) {
            case MAP:
                unit.addImport(LinkedHashMap.class);
                return newObjectBlockStmt(
                    field,
                    byClassName(LinkedHashMap.class, field));
            case SET:
                unit.addImport(LinkedHashSet.class);
                return newObjectBlockStmt(
                    field,
                    byClassName(LinkedHashSet.class, field));
            case LIST:
                unit.addImport(ArrayList.class);
                return newObjectBlockStmt(
                    field,
                    byClassName(ArrayList.class, field));
            case CONFIG:
                return newObjectBlockStmt(
                    field,
                    new ClassOrInterfaceType(
                        null,
                        new SimpleName(builderClass(field)),
                        field.returnType().asClassOrInterfaceType()
                            .getTypeArguments().orElse(
                            null)));
            default:
                return expressionSetBlockStmt(field, new NameExpr("value"));
        }
    }

    protected static BlockStmt newObjectBlockStmt(
        final ConfigField field,
        final ClassOrInterfaceType type)
    {
        return expressionSetBlockStmt(
            field,
            new ObjectCreationExpr(
                null,
                type,
                new NodeList<>(new NameExpr("value"))));
    }

    protected static BlockStmt expressionSetBlockStmt(
        final ConfigField field,
        final Expression expression)
    {
        return new BlockStmt()
            .addStatement(
                new ExpressionStmt(
                    new AssignExpr(
                        new FieldAccessExpr(
                            new ThisExpr(), field.name()),
                        expression,
                        AssignExpr.Operator.ASSIGN)))
            .addStatement(new ReturnStmt(new MethodCallExpr("self")));
    }

    private static ClassOrInterfaceType byClassName(
        final Class<?> clazz,
        final ConfigField field)
    {
        return new ClassOrInterfaceType(
            null,
            new SimpleName(clazz.getSimpleName()),
            field.returnType().asClassOrInterfaceType()
                .getTypeArguments().get());
    }

    public File file() {
        return builderFile;
    }
}
