package ru.yandex.config.generator;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.ArrayList;
import java.util.Collection;
import java.util.EnumSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Scanner;
import java.util.Set;
import java.util.function.Supplier;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.ImportDeclaration;
import com.github.javaparser.ast.Modifier;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.BodyDeclaration;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.printer.PrettyPrinterConfiguration;

public abstract class AbstractConfigGenerator
    implements Supplier<CompilationUnit>
{
    protected static final ImportDeclaration LIST_IMPORT =
        new ImportDeclaration(
            "java.util.ArrayList",
            false,
            false);

    protected static final ImportDeclaration SET_IMPORT =
        new ImportDeclaration(
            "java.util.LinkedHashSet",
            false,
            false);

    protected static final ImportDeclaration MAP_IMPORT =
        new ImportDeclaration(
            "java.util.LinkedHashMap",
            false,
            false);

    protected static final EnumSet<Modifier> FINAL_MODIFIER =
        EnumSet.of(Modifier.FINAL);

    private static final PrettyPrinterConfiguration PRINTER_CONFIG =
        new PrettyPrinterConfiguration();
    static {
        PRINTER_CONFIG.setColumnAlignParameters(true);
        PRINTER_CONFIG.setOrderImports(true);
    }


    protected final Config config;
    protected final String configName;

    public AbstractConfigGenerator(
        final Config config,
        final String configName)
    {
        this.config = config;
        this.configName = configName;
    }

    protected void process() throws IOException {
        String fileName = configName + ".java";
        File out = new File(config.directory(), fileName);
        if (out.exists()) {
            File tmp = new File("/tmp" , fileName);
            Files.copy(Paths.get(out.toURI()), new FileOutputStream(tmp));
        }

        Files.write(
            Paths.get(out.toURI()),
            toString(get()).getBytes(StandardCharsets.UTF_8));
    }

    private static String toString(final CompilationUnit unit) {
        CheckStylePrettyPrintVisitor visitor =
            new CheckStylePrettyPrintVisitor(PRINTER_CONFIG);
        unit.accept(visitor, null);
        return visitor.getSource();
    }

    protected void handleCollectionsImport(
        final Collection<ImportDeclaration> imports,
        final Collection<ImportDeclaration> importsToAdd)
    {
        Iterator<ImportDeclaration> importsIter = imports.iterator();
        while (importsIter.hasNext()) {
            String name = importsIter.next().getNameAsString();
            if ("java.util.List".equalsIgnoreCase(name)) {
                importsIter.remove();
                importsToAdd.add(LIST_IMPORT);
            }

            if ("java.util.Set".equalsIgnoreCase(name)) {
                importsIter.remove();
                importsToAdd.add(SET_IMPORT);
            }

            if ("java.util.Map".equalsIgnoreCase(name)) {
                importsIter.remove();
                importsToAdd.add(MAP_IMPORT);
            }
        }
    }

    protected boolean askUser(final String question) {
        Scanner reader = new Scanner(System.in);  // Reading from System.in
        System.out.println(question);
        String answer = reader.next();

        return answer.equalsIgnoreCase("yes")
            || answer.equalsIgnoreCase("Y");
    }

    protected ImportDeclaration handleImport(
        final String expecting,
        final String replacement,
        final boolean delete,
        final Collection<ImportDeclaration> decs)
    {
        Iterator<ImportDeclaration> iterator = decs.iterator();
        while (iterator.hasNext()) {
            ImportDeclaration dec = iterator.next();
            String name = dec.getNameAsString();
            String[] split = name.split("\\.");
            if (split[split.length - 1].equalsIgnoreCase(expecting)) {
                if (delete) {
                    iterator.remove();
                }

                if (replacement != null) {
                    split[split.length - 1] = replacement;
                    return new ImportDeclaration(
                        String.join(".", split),
                        false,
                        false);
                }
            }
        }

        return null;
    }

    protected abstract String fieldConfigString(final ConfigField field);

    protected void handleConfigsImports(
        final Config config,
        final Collection<ImportDeclaration> imports,
        final boolean deleteInterfaces)
    {
        ImportDeclaration dec;
        for (ConfigField field: config.fields()) {
            switch (field.fieldType()) {
                case CONFIG:
                    String replacement = fieldConfigString(field);
                    dec = handleImport(
                        field.returnType().asString(),
                        replacement,
                        deleteInterfaces,
                        imports);
                    if (dec != null) {
                        imports.add(dec);
                    }

                    break;
                default:
                    break;
            }
        }
    }

    protected void adjustImports(final CompilationUnit unit) {
        List<ImportDeclaration> decsList = new ArrayList<>(unit.getImports());
        Iterator<ImportDeclaration> iterator = decsList.iterator();
        Set<String> uniqueDecs = new LinkedHashSet<>();
        while (iterator.hasNext()) {
            ImportDeclaration dec = iterator.next();
            if (dec.isAsterisk() || dec.isStatic()) {
                continue;
            }

            uniqueDecs.add(dec.getNameAsString());
            iterator.remove();
        }

        for (String dec: uniqueDecs) {
            decsList.add(
                new ImportDeclaration(dec, false, false));
        }

        unit.setImports(new NodeList<>(decsList));
    }

    protected void adjustNodes(final ClassOrInterfaceDeclaration dec) {
        List<FieldDeclaration> fields = dec.getFields();
        List<ConstructorDeclaration> cntrs = dec.getConstructors();
        List<MethodDeclaration> mthds = dec.getMethods();

        NodeList<BodyDeclaration<?>> list = new NodeList<>();
        list.addAll(fields);
        list.addAll(cntrs);
        list.addAll(mthds);

        dec.setMembers(list);
    }
}
