package ru.yandex.config.generator;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.ImportDeclaration;
import com.github.javaparser.ast.PackageDeclaration;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.ast.type.ClassOrInterfaceType;
import com.github.javaparser.ast.type.Type;

public class Config {
    private final ClassOrInterfaceDeclaration configInterface;
    private final CompilationUnit unit;
    private final String name;
    private final File directory;
    private final PackageDeclaration packageDeclaration;
    private final List<ConfigField> fields;
    private final String extendsName;
    private final String extendsPackage;

    public Config(
        final CompilationUnit unit,
        final String name,
        final File directory)
        throws IOException
    {
        this.unit = unit;
        this.name = name;
        this.directory = directory;
        Optional<ClassOrInterfaceDeclaration> optionalClass =
            unit.getInterfaceByName(name);
        if (!optionalClass.isPresent()) {
            throw new IOException("No interface declared for " + name);
        }

        configInterface = optionalClass.get();
        Optional<PackageDeclaration> packageOptional =
            unit.getPackageDeclaration();
        if (!packageOptional.isPresent()) {
            throw new IOException("Default package not allowe");
        }

        this.packageDeclaration = packageOptional.get();

        fields = new ArrayList<>();

        for (MethodDeclaration dec: configInterface.getMethods()) {
            Type returnType = dec.getType();
            String methodName = dec.getNameAsString();
            FieldType fieldType = detectFieldType(returnType);
            fields.add(new ConfigField(fieldType, methodName, returnType));
        }

        if (configInterface.getExtendedTypes().size() == 1) {
            extendsName =
                configInterface.getExtendedTypes().get(0).getNameAsString();
            String extendsPackage = packageDeclaration.getNameAsString();
            for (ImportDeclaration dec: unit.getImports()) {
                String packName = dec.getNameAsString();
                if (packName.contains(extendsName)) {
                    int dotIndex = packName.lastIndexOf('.');
                    if (dotIndex >= 0) {
                        extendsPackage = packName.substring(0, dotIndex);
                    }

                    break;
                }
            }

            this.extendsPackage = extendsPackage;
        } else {
            extendsName = null;
            extendsPackage = null;
        }
    }

    public String defaultsName() {
        return name + "Defaults";
    }

    public ClassOrInterfaceType defaultsType() {
        return new ClassOrInterfaceType(null, defaultsName());
    }

    public String immutableName() {
        return "Immutable" + name;
    }

    public ClassOrInterfaceType immutableType() {
        return new ClassOrInterfaceType(null, immutableName());
    }

    public String builderName() {
        return name + "Builder";
    }

    public String abstractBuilderName() {
        return  "Abstract" + name + "Builder";
    }

    public ClassOrInterfaceType builderType() {
        return new ClassOrInterfaceType(null, builderName());
    }

    public PackageDeclaration getPackage() {
        return packageDeclaration;
    }

    public CompilationUnit unit() {
        return unit;
    }

    public String name() {
        return name;
    }

    public File directory() {
        return directory;
    }

    public ClassOrInterfaceDeclaration configInterface() {
        return configInterface;
    }

    public List<ConfigField> fields() {
        return fields;
    }

    private static FieldType detectFieldType(final Type returnType) {
        if (returnType.isClassOrInterfaceType()) {
            if (returnType.asString().endsWith("Config")
                && !returnType.asString().equalsIgnoreCase(
                    "IniConfig"))
            {
                return FieldType.CONFIG;
            }

            if ("Set".equalsIgnoreCase(returnType.asClassOrInterfaceType()
                .getName().asString()))
            {
                return FieldType.SET;
            }

            if ("List".equalsIgnoreCase(
                returnType.asClassOrInterfaceType().getName().asString()))
            {
                return FieldType.LIST;
            }

            if ("Map".equalsIgnoreCase(
                returnType.asClassOrInterfaceType().getName().asString()))
            {
                return FieldType.MAP;
            }
        }

        return FieldType.IMMUTABLE;
    }

    public String extendsName() {
        return extendsName;
    }

    public String extendsPackage() {
        return extendsPackage;
    }
}
