package ru.yandex.config.generator;

import java.io.File;
import java.io.IOException;
import java.util.Optional;

import com.github.javaparser.JavaParser;
import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.Modifier;
import com.github.javaparser.ast.NodeList;
import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.Parameter;
import com.github.javaparser.ast.expr.FieldAccessExpr;
import com.github.javaparser.ast.expr.MarkerAnnotationExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.expr.SimpleName;
import com.github.javaparser.ast.expr.ThisExpr;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExplicitConstructorInvocationStmt;
import com.github.javaparser.ast.stmt.ReturnStmt;
import com.github.javaparser.ast.type.ClassOrInterfaceType;

public class ConfigBuilderGenerator extends AbstractConfigGenerator {
    private final ClassOrInterfaceDeclaration configClass;
    private final CompilationUnit unit;
    private final File configFile;
    private final ClassOrInterfaceType interfaceType;

    public ConfigBuilderGenerator(
        final Config config)
        throws IOException
    {
        super(config, config.builderName());
        this.interfaceType =
            new ClassOrInterfaceType(null, config.name());
        this.configFile =
            new File(config.directory(), configName + ".java");

        if (configFile.exists()) {
            unit = JavaParser.parse(configFile);
            Optional<ClassOrInterfaceDeclaration> optional =
                unit.getClassByName(configName);
            if (!optional.isPresent()) {
                throw new IOException(configFile.getAbsolutePath()
                    + " exists but do not contain class inside "
                    + configName);
            }

            configClass = optional.get();
        } else {
            unit = new CompilationUnit(
                config.getPackage(),
                new NodeList<>(),
                new NodeList<>(),
                null);

            configClass= unit.addClass(configName);

            configClass.addExtendedType(new ClassOrInterfaceType(
                null,
                new SimpleName("Abstract" + config.name() + "Builder"),
                new NodeList<>(
                    new ClassOrInterfaceType(null, configName))));
            unit.addImport("ru.yandex.parser.config.ConfigException");
            unit.addImport("ru.yandex.parser.config.IniConfig");

            addEmptyConstructor();
            addConfigConstructor();
            addIniConfigConstructor();
            addFullConstructor();
            addBuildMethod();
            addSelfMethod();
        }
    }

    protected void addEmptyConstructor() {
        configClass.addConstructor(Modifier.PUBLIC).setBody(
            new BlockStmt(
                NodeList.nodeList(
                    new ExplicitConstructorInvocationStmt(
                        true,
                        null,
                        NodeList.nodeList(
                            new FieldAccessExpr(
                                new NameExpr(config.defaultsName()),
                            "INSTANCE"))))));
    }

    protected void addConfigConstructor() {
        configClass.addConstructor(Modifier.PUBLIC)
            .addParameter(
                new Parameter(
                    FINAL_MODIFIER,
                    interfaceType,
                    new SimpleName("config")))
            .setBody(
                new BlockStmt(
                    NodeList.nodeList(
                        new ExplicitConstructorInvocationStmt(
                            false,
                            null,
                            NodeList.nodeList(new NameExpr("config"))))));
    }

    protected void addIniConfigConstructor() {
        BlockStmt body = new BlockStmt(
            NodeList.nodeList(
                new ExplicitConstructorInvocationStmt(
                    true,
                    null,
                    NodeList.nodeList(
                        new NameExpr("config"),
                        new FieldAccessExpr(
                            new NameExpr(config.defaultsName()),
                            "INSTANCE")))));
        configClass.addConstructor(Modifier.PUBLIC)
            .addParameter(
                new Parameter(
                    FINAL_MODIFIER,
                new ClassOrInterfaceType(
                    null,
                    "IniConfig"),
                new SimpleName("config")))
            .addThrownException(
                new ClassOrInterfaceType(null, "ConfigException"))
            .setBody(body);
    }

    protected void addFullConstructor() {
        BlockStmt body = new BlockStmt(
            NodeList.nodeList(
                new ExplicitConstructorInvocationStmt(
                    false,
                    null,
                    NodeList.nodeList(
                        new NameExpr("config"),
                        new NameExpr("defaults")))));
        configClass.addConstructor(Modifier.PUBLIC)
            .addParameter(
                new Parameter(
                    FINAL_MODIFIER,
                new ClassOrInterfaceType(
                    null,
                    "IniConfig"),
                new SimpleName("config")))
            .addParameter(
                new Parameter(
                    FINAL_MODIFIER,
                    interfaceType,
                    new SimpleName("defaults")))
            .addThrownException(
                new ClassOrInterfaceType(null, "ConfigException"))
            .setBody(body);
    }

    protected void addBuildMethod() {
        configClass.addMethod("build", Modifier.PUBLIC)
            .setType(config.immutableType())
            .addThrownException(
                new ClassOrInterfaceType(null, "ConfigException"))
            .setBody(
                new BlockStmt(NodeList.nodeList(
                    new ReturnStmt(
                        new ObjectCreationExpr(
                            null,
                            config.immutableType(),
                            NodeList.nodeList(new ThisExpr()))))));
    }

    protected void addSelfMethod() {
        configClass
            .addMethod("self", Modifier.PROTECTED)
            .addAnnotation(new MarkerAnnotationExpr("Override"))
            .setType(config.builderType())
            .setBody(
                new BlockStmt(NodeList.nodeList(new ReturnStmt(new ThisExpr()
                ))));
    }

    @Override
    protected String fieldConfigString(final ConfigField field) {
        return field.builderType().asString();
    }

    @Override
    public CompilationUnit get() {
        return unit;
    }
}
