package ru.yandex.config.generator;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.github.javaparser.JavaParser;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.ImportDeclaration;
import com.github.javaparser.ast.Modifier;
import com.github.javaparser.ast.NodeList;

import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.ast.body.VariableDeclarator;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.FieldAccessExpr;
import com.github.javaparser.ast.expr.MarkerAnnotationExpr;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.NullLiteralExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ReturnStmt;
import com.github.javaparser.ast.type.ClassOrInterfaceType;

public class DefaultsConfigGenerator extends AbstractConfigGenerator {
    private final ClassOrInterfaceDeclaration configClass;
    private final CompilationUnit unit ;
    private final File configFile;
    private final String extendsClass;

    public DefaultsConfigGenerator(
        final Config config)
        throws IOException
    {
        super(config, config.defaultsName());
        this.configFile =
            new File(config.directory(), configName + ".java");
        if (config.extendsName() != null) {
            extendsClass =
                config.extendsName() + "Defaults";
        } else {
            extendsClass = null;
        }

        if (configFile.exists()) {
            unit = JavaParser.parse(configFile);
            Optional<ClassOrInterfaceDeclaration> optional =
                unit.getClassByName(configName);
            if (!optional.isPresent()) {
                throw new IOException(configFile.getAbsolutePath()
                    + " exists but do not contain class inside "
                    + configName);
            }

            configClass = optional.get();
        } else {
            unit = new CompilationUnit(
                config.getPackage(),
                new NodeList<>(config.unit().getImports()),
                new NodeList<>(),
                null);

            configClass= unit.addClass(configName);

            if (extendsClass != null) {
                configClass.addExtendedType(extendsClass);
            }

            configClass.addImplementedType(config.name());
        }
    }

    @Override
    protected String fieldConfigString(final ConfigField field) {
        return field.defaultsType().asString();
    }

    private void handleConstrustor() {
        boolean declared = false;
        for (ConstructorDeclaration md: configClass.getConstructors()) {
            if (md.getParameters().size() == 0) {
                declared = true;
                break;
            }
        }

        if (!declared) {
            configClass.addConstructor(Modifier.PROTECTED).setBody(
                new BlockStmt());
        }
    }

    private void handleFields() {
        Optional<FieldDeclaration> fd = configClass.getFieldByName("INSTANCE");
        if (!fd.isPresent()) {
            configClass.addFieldWithInitializer(
                configName,
                "INSTANCE",
                new ObjectCreationExpr(
                    null,
                    new ClassOrInterfaceType(null, configName),
                    NodeList.nodeList()),
                Modifier.PUBLIC,
                Modifier.STATIC,
                Modifier.FINAL);
        }
    }

    private void handleImports() {
        Collection<ImportDeclaration> imports = unit.getImports();
        handleConfigsImports(config, imports, false);
        // extends class import
        ImportDeclaration dec =
            handleImport(config.extendsName(), extendsClass, true, imports);
        if (dec != null) {
            imports.add(dec);
        }

        unit.setImports(new NodeList<>(imports));
        adjustImports(unit);
    }

    private void handleMethods() {
        Map<String, ConfigField> fields = config.fields().stream().collect(
            Collectors.toMap(ConfigField::name, Function.identity()));

        Iterator<MethodDeclaration> methodIterator =
            configClass.getMethods().iterator();

        Set<String> fieldsWithGetMethods = new LinkedHashSet<>();

        while (methodIterator.hasNext()) {
            MethodDeclaration md = methodIterator.next();

            if (md.getModifiers().contains(Modifier.PRIVATE)) {
                continue;
            }

            String typeName = md.getTypeAsString();
            if (!fields.containsKey(md.getNameAsString())) {
                if (md.getParameters().size() == 0
                    && !md.getType().isVoidType())
                {
                    if (askUser(
                        "Y/Yes remove method " + md.getNameAsString() + " from "
                            + configName))
                    {
                        configClass.remove(md);
                    }
                } else if (md.getParameters().size() == 1
                    && typeName.length() == 1)
                {
                    if (askUser(
                        "Y/Yes remove method " + md.getNameAsString() + " from "
                            + configName))
                    {
                        configClass.remove(md);
                    }
                }

                continue;
            }

            if (md.getParameters().size() == 0) {
                fieldsWithGetMethods.add(md.getNameAsString());
            }
        }

        for (ConfigField field: config.fields()) {
            if (!fieldsWithGetMethods.contains(field.name())) {
                MethodDeclaration methodDeclaration =
                    configClass.addMethod(field.name(), Modifier.PUBLIC);
                methodDeclaration.addAnnotation(
                    new MarkerAnnotationExpr("Override"));
                methodDeclaration.setType(field.returnType());
                Expression expression = new NullLiteralExpr();
                switch (field.fieldType()) {
                    case CONFIG:
                        expression =
                            new FieldAccessExpr(
                                new NameExpr(field.defaultsType().asString()),
                                "INSTANCE");
                        break;
                    case SET:
                        unit.addImport(Collections.class);
                        expression = new MethodCallExpr(
                            new NameExpr("Collections"),
                            "emptySet");
                        break;
                    case LIST:
                        unit.addImport(Collections.class);
                        expression = new MethodCallExpr(
                            new NameExpr("Collections"),
                            "emptyList");
                        break;
                    case MAP:
                        unit.addImport(Collections.class);
                        expression = new MethodCallExpr(
                            new NameExpr("Collections"),
                            "emptyMap");
                        break;
                    default:
                        break;
                }

                methodDeclaration.setBody(
                    new BlockStmt().addStatement(
                        new ReturnStmt(expression)));
            }
        }
    }

    @Override
    public CompilationUnit get() {
        handleImports();
        handleFields();
        handleConstrustor();
        handleMethods();
        adjustImports(unit);
        return unit;
    }
}
