package ru.yandex.config.generator;

import java.io.File;
import java.io.IOException;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.github.javaparser.JavaParser;

import com.github.javaparser.ast.CompilationUnit;
import com.github.javaparser.ast.ImportDeclaration;
import com.github.javaparser.ast.Modifier;
import com.github.javaparser.ast.NodeList;

import com.github.javaparser.ast.body.ClassOrInterfaceDeclaration;
import com.github.javaparser.ast.body.ConstructorDeclaration;
import com.github.javaparser.ast.body.FieldDeclaration;
import com.github.javaparser.ast.body.MethodDeclaration;
import com.github.javaparser.ast.body.Parameter;

import com.github.javaparser.ast.expr.AssignExpr;
import com.github.javaparser.ast.expr.Expression;
import com.github.javaparser.ast.expr.FieldAccessExpr;
import com.github.javaparser.ast.expr.MarkerAnnotationExpr;
import com.github.javaparser.ast.expr.MethodCallExpr;
import com.github.javaparser.ast.expr.NameExpr;
import com.github.javaparser.ast.expr.ObjectCreationExpr;
import com.github.javaparser.ast.expr.SimpleName;
import com.github.javaparser.ast.expr.ThisExpr;
import com.github.javaparser.ast.stmt.BlockStmt;
import com.github.javaparser.ast.stmt.ExplicitConstructorInvocationStmt;
import com.github.javaparser.ast.stmt.ExpressionStmt;
import com.github.javaparser.ast.stmt.ReturnStmt;
import com.github.javaparser.ast.stmt.Statement;
import com.github.javaparser.ast.type.ClassOrInterfaceType;

public class ImmutableConfigGenerator extends AbstractConfigGenerator {
    private final ClassOrInterfaceDeclaration configClass;
    private final CompilationUnit unit ;
    private final File configFile;

    public ImmutableConfigGenerator(
        final Config config)
        throws IOException
    {
        super(config, config.immutableName());

        this.configFile =
            new File(config.directory(), configName + ".java");
        if (configFile.exists()) {
            unit = JavaParser.parse(configFile);
            Optional<ClassOrInterfaceDeclaration> optional =
                unit.getClassByName(configName);
            if (!optional.isPresent()) {
                throw new IOException(configFile.getAbsolutePath()
                    + " exists but do not contain class inside "
                    + configName);
            }

            configClass = optional.get();
        } else {
            unit = new CompilationUnit(
                config.getPackage(),
                new NodeList<>(config.unit().getImports()),
                new NodeList<>(),
                null);

            configClass= unit.addClass(configName);

            if (config.extendsName() != null) {
                String extendsClass =
                    "Immutable" + config.extendsName();
                configClass.addExtendedType(extendsClass);
                unit.addImport(
                    config.extendsPackage() + '.' + extendsClass);
            }

            configClass.addImplementedType(config.name());
        }
    }

    protected void handleImports() {
        Collection<ImportDeclaration> imports = unit.getImports();
        imports.add(
            new ImportDeclaration(
                "ru.yandex.parser.config.ConfigException",
                false,
                false));

        handleConfigsImports(config, imports, true);
        if (config.extendsName() != null) {
            handleImport(config.extendsName(), null, true, imports);
        }

        unit.setImports(new NodeList<>(imports));
        adjustImports(unit);
    }

    protected void handleConstrustor() {
        ConstructorDeclaration constructor = null;
        for (ConstructorDeclaration md: configClass.getConstructors()) {
            if (md.getParameters().size() == 1) {
                String first = md.getParameter(0).getType().asString();
                if (config.name().equalsIgnoreCase(first)) {
                    constructor = md.asConstructorDeclaration();
                    break;
                }
            }
        }

        NodeList<Statement> constrStmts = new NodeList<>();
        if (constructor == null) {
            constructor =
                configClass.addConstructor(Modifier.PUBLIC);
            constructor.addParameter(
                new Parameter(
                    FINAL_MODIFIER,
                    new ClassOrInterfaceType(
                        null,
                        config.name()),
                    new SimpleName("config")));
            constructor.addThrownException(
                new ClassOrInterfaceType(
                    null,
                    "ConfigException"));

            constrStmts.add(
                new ExplicitConstructorInvocationStmt(
                    false,
                    null,
                    NodeList.nodeList(new NameExpr("config"))));
        } else {
            constrStmts = constructor.getBody().asBlockStmt().getStatements();
        }

        Set<String> initedFields = new LinkedHashSet<>();
        for (Statement statement: constrStmts) {
            if (statement.isExpressionStmt()) {
                Expression expression =
                    statement.asExpressionStmt().getExpression();
                if (expression.isAssignExpr()) {
                    Expression target = expression.asAssignExpr().getTarget();
                    if (target.isFieldAccessExpr()) {
                        initedFields.add(
                            target.asFieldAccessExpr().getNameAsString());
                    }
                }
            }
        }

        for (ConfigField field: config.fields()) {
            if (initedFields.contains(field.name())) {
                continue;
            }

            NodeList<Expression> args;
            switch (field.fieldType()) {
                case CONFIG:
                    args =
                        new NodeList<>(
                            new MethodCallExpr(
                                new NameExpr("config"),
                                new SimpleName(field.name())));
                    constrStmts.add(
                        new ExpressionStmt(
                            new AssignExpr(
                                new FieldAccessExpr(
                                    new ThisExpr(),
                                    field.name()),
                                new ObjectCreationExpr(
                                    null,
                                    new ClassOrInterfaceType(
                                        null,
                                        field.immutableType().asString()),
                                    args),
                                AssignExpr.Operator.ASSIGN)));
                    break;
                case SET:
                    args = new NodeList<>(
                        new MethodCallExpr(
                            new NameExpr("config"),
                            new SimpleName(field.name())));
                    constrStmts.add(
                        new ExpressionStmt(
                            new AssignExpr(
                                new FieldAccessExpr(
                                    new ThisExpr(),
                                    field.name()),
                                new MethodCallExpr(
                                    null,
                                    "Collections.unmodifiableSet",
                                    args),
                                AssignExpr.Operator.ASSIGN)));
                    unit.addImport(Collections.class);
                    break;
                case LIST:
                    args = new NodeList<>(
                        new MethodCallExpr(
                            new NameExpr("config"),
                            new SimpleName(field.name())));
                    constrStmts.add(
                        new ExpressionStmt(
                            new AssignExpr(
                                new FieldAccessExpr(
                                    new ThisExpr(),
                                    field.name()),
                                new MethodCallExpr(
                                    null,
                                    "Collections.unmodifiableList",
                                    args),
                                AssignExpr.Operator.ASSIGN)));
                    unit.addImport(Collections.class);
                    break;
                case MAP:
                    args = new NodeList<>(
                        new MethodCallExpr(
                            new NameExpr("config"),
                            new SimpleName(field.name())));
                    constrStmts.add(
                        new ExpressionStmt(
                            new AssignExpr(
                                new FieldAccessExpr(
                                    new ThisExpr(),
                                    field.name()),
                                new MethodCallExpr(
                                    null,
                                    "Collections.unmodifiableMap",
                                    args),
                                AssignExpr.Operator.ASSIGN)));
                    unit.addImport(Collections.class);
                    break;
                case IMMUTABLE:
                    constrStmts.add(
                        new ExpressionStmt(
                            new AssignExpr(
                                new FieldAccessExpr(
                                    new ThisExpr(),
                                    field.name()),
                                new MethodCallExpr(
                                    new NameExpr("config"),
                                    new SimpleName(field.name())),
                                AssignExpr.Operator.ASSIGN)));
                    break;
            }
        }

        constructor.setBody(new BlockStmt(constrStmts));
    }

    @Override
    protected String fieldConfigString(final ConfigField field) {
        return field.immutableType().asString();
    }

    protected void handleMethods() {
        Map<String, ConfigField> fields = config.fields().stream().collect(
            Collectors.toMap(ConfigField::name, Function.identity()));

        Iterator<MethodDeclaration> methodIterator =
            configClass.getMethods().iterator();

        Set<String> fieldsWithGetMethods = new LinkedHashSet<>();

        while (methodIterator.hasNext()) {
            MethodDeclaration md = methodIterator.next();

            if (md.getModifiers().contains(Modifier.PRIVATE)) {
                continue;
            }

            String typeName = md.getTypeAsString();
            if (!fields.containsKey(md.getNameAsString())) {
                if (md.getParameters().size() == 0
                    && !md.getType().isVoidType())
                {
                    if (askUser(
                        "Y/Yes remove method " + md.getNameAsString() + " from "
                            + configName))
                    {
                        configClass.remove(md);
                    }
                } else if (md.getParameters().size() == 1
                    && typeName.length() == 1)
                {
                    if (askUser(
                        "Y/Yes remove method " + md.getNameAsString() + " from "
                            + configName))
                    {
                        configClass.remove(md);
                    }
                }

                continue;
            }

            if (md.getParameters().size() == 0) {
                fieldsWithGetMethods.add(md.getNameAsString());
            }
        }

        for (ConfigField field: config.fields()) {
            if (!fieldsWithGetMethods.contains(field.name())) {
                MethodDeclaration methodDeclaration =
                    configClass.addMethod(field.name(), Modifier.PUBLIC);
                methodDeclaration.addAnnotation(
                    new MarkerAnnotationExpr("Override"));
                methodDeclaration.setType(field.immutableType());
                methodDeclaration.setBody(
                    new BlockStmt().addStatement(
                        new ReturnStmt(
                            new NameExpr(field.name()))));
            }
        }
    }

    protected void handleClassFields() {
        for (ConfigField field: config.fields()) {
            Optional<FieldDeclaration> optional =
                configClass.getFieldByName(field.name());
            if (!optional.isPresent()) {
                configClass.addField(
                    field.immutableType(),
                    field.name(),
                    Modifier.PRIVATE,
                    Modifier.FINAL);
            }
        }
    }

    @Override
    public CompilationUnit get() {
        handleImports();
        handleClassFields();
        handleConstrustor();
        handleMethods();
        adjustImports(unit);
        adjustNodes(configClass);
        return unit;
    }
}
