package ru.yandex.tools.jacoco;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;

import org.jacoco.core.analysis.Analyzer;
import org.jacoco.core.analysis.CoverageBuilder;
import org.jacoco.core.tools.ExecFileLoader;
import org.jacoco.report.FileMultiReportOutput;
import org.jacoco.report.IReportVisitor;
import org.jacoco.report.InputStreamSourceFileLocator;
import org.jacoco.report.csv.CSVFormatter;
import org.jacoco.report.html.HTMLFormatter;

public final class Coverage {
    private static final int TAB_SIZE = 4;

    private Coverage() {
    }

    private static void load(final ExecFileLoader loader, final File dir)
        throws IOException
    {
        if (dir.exists()) {
            File[] files = dir.listFiles();
            if (files != null) {
                for (File file: files) {
                    if (!file.isDirectory()
                        && file.getCanonicalPath().endsWith(".exec"))
                    {
                        loader.load(file);
                    }
                }
            }
        }
    }

    private static void analyze(final Analyzer analyzer, final File file)
        throws IOException
    {
        if (file.exists()) {
            if (file.isDirectory()) {
                if (!file.getCanonicalPath().endsWith("/generated")) {
                    File[] files = file.listFiles();
                    if (files != null) {
                        for (File f: files) {
                            analyze(analyzer, f);
                        }
                    }
                }
            } else {
                analyzer.analyzeAll(file);
            }
        }
    }

    public static void main(final String... args) throws Exception {
        ExecFileLoader loader = new ExecFileLoader();
        for (String arg: args) {
            load(loader, new File("build/report/" + arg));
        }
        CoverageBuilder builder = new CoverageBuilder();
        Analyzer analyzer =
            new Analyzer(loader.getExecutionDataStore(), builder);
        for (String arg: args) {
            analyze(analyzer, new File("build/" + arg + "/main/classes/ru"));
        }
        try (FileOutputStream out =
                new FileOutputStream("build/report/coverage.csv"))
        {
            IReportVisitor[] visitors = {
                new HTMLFormatter().createVisitor(
                    new FileMultiReportOutput(
                        new File("build/report/coverage"))),
                new CSVFormatter().createVisitor(out)
            };
            for (IReportVisitor visitor: visitors) {
                visitor.visitInfo(
                    loader.getSessionInfoStore().getInfos(),
                    loader.getExecutionDataStore().getContents());
                visitor.visitBundle(
                    builder.getBundle("Code Coverage"),
                    new FileLocator(args));
                visitor.visitEnd();
            }
        }
        loader.save(new File("build/report/coverage/jacoco.exec"), false);
    }

    private static class FileLocator extends InputStreamSourceFileLocator {
        private final String[] projects;

        FileLocator(final String[] projects) {
            super("utf-8", TAB_SIZE);
            this.projects = projects;
        }

        @Override
        protected InputStream getSourceStream(final String path)
            throws IOException
        {
            for (String project: projects) {
                File file = new File("src/" + project + "/main/java", path);
                if (file.exists()) {
                    return new FileInputStream(file);
                }
            }
            return null;
        }
    }
}
