package ru.yandex.search.tts.gorgophone;

import java.io.IOException;

import java.text.ParseException;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriterBase;

import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.result.SearchDocument;

public abstract class AbstractRecordContext<
    T extends AbstractRecordContext<T, U>,
    U extends LuceneFields>
    implements JsonValue
{
    public static final String UID = "uid";
    public static final String CGI_CREATE_TIME = "create-time";
    public static final String CGI_CHANGE_TIME = "change-time";
    public static final String CGI_DELETE_TIME = "delete-time";
    public static final String CGI_PROCESSING_TIME = "processing-time";
    public static final String CGI_PROCESSOR_ID = "processor-id";

    private static final String LESS = " > ";

    private final String uid;
    private final long createTime;
    private final long changeTime;
    private final long deleteTime;
    private final long processingTime;
    private final String processorId;
    private final String pk;
    private final U fieldMap;
    private final boolean current;
    private final long queueId;

    public AbstractRecordContext(
        final CgiParams params,
        final U fieldMap)
        throws BadRequestException
    {
        this.fieldMap = fieldMap;
        this.queueId = -1;

        final long currentTime = System.currentTimeMillis();
        Long createTime = params.getLong(CGI_CREATE_TIME, null);
        if (createTime == null) {
            createTime = currentTime;
        }
        this.createTime = createTime;
        changeTime = params.getLong(CGI_CHANGE_TIME, currentTime);
        deleteTime = params.getLong(CGI_DELETE_TIME, Long.MAX_VALUE);
        processingTime = params.getLong(CGI_PROCESSING_TIME, currentTime);

        if (changeTime < createTime) {
            throw new BadRequestException(
                "CreateTime is bigger than ChangeTime: " + createTime
                    + LESS + changeTime);
        }
        if (deleteTime < createTime) {
            throw new BadRequestException(
                "DeleteTime is bigger than CreateTime: " + deleteTime
                    + LESS + createTime);
        }
        if (deleteTime < changeTime) {
            throw new BadRequestException(
                "DeleteTime is bigger than ChangeTime: " + deleteTime
                    + LESS + changeTime);
        }

        uid = params.getString(UID);
        processorId = params.getString(CGI_PROCESSOR_ID);
        pk = Long.toString(queueId);
        current = false;
    }

    public AbstractRecordContext(
        final CgiParams params,
        final long queueId,
        final U fieldMap)
        throws BadRequestException
    {
        this.queueId = queueId;
        this.fieldMap = fieldMap;
        uid = params.getString(UID, null);
        createTime = params.getLong(CGI_CREATE_TIME);
        changeTime = params.getLong(CGI_CHANGE_TIME);
        deleteTime = params.getLong(CGI_DELETE_TIME);
        processingTime = params.getLong(CGI_PROCESSING_TIME);
        processorId = params.getString(CGI_PROCESSOR_ID);
        pk = Long.toString(queueId);
        current = false;
    }

    public AbstractRecordContext(
        final SearchDocument doc,
        final U fieldMap)
        throws ParseException
    {
        this.fieldMap = fieldMap;
        try {
            queueId = Long.parseLong(doc.attrs().get(fieldMap.queueId()));
            uid = doc.attrs().get(fieldMap.uid());
            createTime =
                Long.parseLong(doc.attrs().get(fieldMap.createTime()));
            changeTime =
                Long.parseLong(doc.attrs().get(fieldMap.changeTime()));
            deleteTime =
                Long.parseLong(doc.attrs().get(fieldMap.deleteTime()));
            processingTime =
                Long.parseLong(doc.attrs().get(fieldMap.processingTime()));
            processorId = doc.attrs().get(fieldMap.processorId());
            pk = Long.toString(queueId);
            current = false;
        } catch (Exception e) {
            ParseException pe = new ParseException(
                "Error parsing loaded document: " + e.getMessage(),
                0);
            pe.addSuppressed(e);
            throw pe;
        }
    }

    protected AbstractRecordContext(
        final AbstractRecordContext<T, U> old,
        final AbstractRecordContext<T, U> current)
    {
        fieldMap = current.fieldMap;
        uid = current.uid;
        processingTime = current.processingTime;
        processorId = current.processorId;
        if (old == null) {
            createTime = current.createTime;
            changeTime = current.changeTime;
            deleteTime = current.deleteTime;
            this.current = true;
            this.queueId = current.queueId;
            this.pk = current.pk;
        } else {
            createTime = Math.min(old.createTime, current.createTime);
            changeTime = current.changeTime;
            deleteTime = Math.min(old.deleteTime, current.deleteTime);
            if (old.changeTime <= current.changeTime) {
                this.current = true;
            } else {
                this.current = false;
            }
            this.queueId = current.queueId;
            this.pk = current.pk;
        }
    }

    public boolean current() {
        return current;
    }

    public abstract String shardingField();

    public abstract void writeFields(
        final JsonWriterBase writer,
        final U fieldMap)
        throws IOException;

    public U fieldMap() {
        return fieldMap;
    }

    public long prefix() {
        long prefix;
        if (uid != null) {
            try {
                prefix = Long.parseLong(uid);
            } catch (java.lang.NumberFormatException e) {
                prefix = shardingField().hashCode();
            }
        } else {
            prefix = shardingField().hashCode();
        }
        return prefix;
    }

    public abstract T adjust(final T other);

    public abstract String id();

    public abstract void appendCgiParams(final QueryConstructor qc)
        throws BadRequestException;

    public String uri(final String basicPath) throws BadRequestException {
        return uriConstructor(basicPath).toString();
    }

    public QueryConstructor uriConstructor(final String basicPath)
         throws BadRequestException
    {
        QueryConstructor qc = new QueryConstructor(basicPath);
        if (uid != null) {
            qc.append(UID, uid);
        }
        qc.append(CGI_CREATE_TIME, createTime);
        qc.append(CGI_CHANGE_TIME, changeTime);
        qc.append(CGI_DELETE_TIME, deleteTime);
        qc.append(CGI_PROCESSING_TIME, processingTime);
        qc.append(CGI_PROCESSOR_ID, processorId);

        appendCgiParams(qc);

        return qc;
    }

    public String pk() {
        return pk + '#' + id();
    }

    public String uid() {
        return uid;
    }

    public long createTime() {
        return createTime;
    }

    public long changeTime() {
        return changeTime;
    }

    public long deleteTime() {
        return deleteTime;
    }

    public long processingTime() {
        return processingTime;
    }

    public String processorId() {
        return processorId;
    }

    @Override
    public void writeValue(final JsonWriterBase writer) throws IOException {
        writer.startObject();
        writer.key(fieldMap.pk());
        writer.value(pk());
        writer.key(fieldMap.id());
        if (current) {
            writer.value("current#" + id());
        } else {
            writer.value(id());
        }
        writer.key(fieldMap.uid());
        writer.value(uid());
        writer.key(fieldMap.current());
        writer.value(current);
        writer.key(fieldMap.queueId());
        writer.value(queueId);
        writer.key(fieldMap.createTime());
        writer.value(createTime);
        writer.key(fieldMap.changeTime());
        writer.value(changeTime);
        writer.key(fieldMap.deleteTime());
        writer.value(deleteTime);
        writer.key(fieldMap.processingTime());
        writer.value(processingTime);
        writer.key(fieldMap.processorId());
        writer.value(processorId);
        writeFields(writer, fieldMap);
        writer.endObject();
    }
}
