package ru.yandex.search.tts.gorgophone;

import java.io.IOException;

import java.nio.charset.StandardCharsets;

import java.text.ParseException;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpStatus;
import org.apache.http.entity.ContentType;

import ru.yandex.http.proxy.AbstractProxySessionCallback;
import ru.yandex.http.proxy.ProxyRequestHandler;
import ru.yandex.http.proxy.ProxySession;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.YandexHeaders;
import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.io.DecodableByteArrayOutputStream;

import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.Utf8JsonWriter;

import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.proxy.SearchResultConsumerFactory;

import ru.yandex.search.request.util.SearchRequestText;

import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public abstract class AddRecordHandlerBase<
    T extends AbstractRecordContext<T, ? extends LuceneFields>>
    implements ProxyRequestHandler
{
    private static final String ADD_IF_NOT_EXISTS = "AddIfNotExists";
    private static final String PREFIX = "prefix";
    private static final String DOCS = "docs";
    private static final String GET = "get";
    private static final String JSON_TYPE = "json-type";
    private static final String TEXT = "text";
    private static final ContentType JSON_CONTENT_TYPE =
        ContentType.create("application/json", StandardCharsets.UTF_8);
    private static final String SEARCH = "/search?";

    private final Gorgophone server;

    public AddRecordHandlerBase(final Gorgophone server) {
        this.server = server;
    }

    @Override
    public void handle(final ProxySession session)
        throws HttpException, IOException
    {
        if (session.request() instanceof HttpEntityEnclosingRequest) {
            handleBatch(session);
        } else {
            handleSingleRecord(session);
        }
    }

    protected abstract T createRecord(
        final ProxySession session,
        final long queueId)
        throws BadRequestException;

    protected abstract T recordFromDocument(final SearchDocument doc)
        throws ParseException;

    private void handleSingleRecord(final ProxySession session)
        throws BadRequestException
    {
        final long queueId =
            session.headers().getInt(YandexHeaders.ZOO_QUEUE_ID);
        final T record = createRecord(
            session,
            queueId);
        final AsyncClient searchClient =
            server.searchClient().adjust(session.context());
        final QueryConstructor request = new QueryConstructor(SEARCH);
        request.append(PREFIX, record.prefix());
        request.append(
            TEXT,
            record.fieldMap().id() + ":current#"
                + SearchRequestText.fullEscape(record.id(), false));
        request.append(GET, record.fieldMap().getFields());
        request.append(JSON_TYPE, "dollar");
        searchClient.execute(
            server.config().searchConfig().host(),
            new BasicAsyncRequestProducerGenerator(request.toString()),
            SearchResultConsumerFactory.OK,
            session.listener().createContextGeneratorFor(searchClient),
            new RecordSearchCallback(session, record));
    }

    private void handleBatch(final ProxySession session) {
        session.response(HttpStatus.SC_NOT_IMPLEMENTED);
    }

    private void updateEntityRecord(
        final T oldRecord,
        final T newRecord,
        final ProxySession session)
        throws IOException
    {
        DecodableByteArrayOutputStream out =
            new DecodableByteArrayOutputStream();
        Utf8JsonWriter writer = JsonType.DOLLAR.create(out);
        writer.startObject();
        writer.key(ADD_IF_NOT_EXISTS);
        writer.value(true);
        writer.key(PREFIX);
        writer.value(newRecord.prefix());
        writer.key(DOCS);
        writer.startArray();
        if (oldRecord == null) {
            writer.value(newRecord.adjust(null));
        } else {
            writer.value(oldRecord.adjust(newRecord));
            writer.value(newRecord.adjust(oldRecord));
        }
        writer.endArray();
        writer.endObject();
        final AsyncClient indexClient =
            server.backendClient().adjustZooHeaders(session.context());
        BasicAsyncRequestProducerGenerator post =
            new BasicAsyncRequestProducerGenerator(
                "/update?property&prefix=" + newRecord.prefix(),
                out.toByteArray(),
                JSON_CONTENT_TYPE);
        indexClient.execute(
            server.config().backendConfig().host(),
            post,
            EmptyAsyncConsumerFactory.ANY_GOOD,
            session.listener().createContextGeneratorFor(indexClient),
            new ResponseCallback(session));
    }

    private class RecordSearchCallback
        extends AbstractProxySessionCallback<SearchResult>
    {
        private final T newRecord;

        RecordSearchCallback(
            final ProxySession session,
            final T newRecord)
        {
            super(session);
            this.newRecord = newRecord;
        }

        @Override
        public void completed(final SearchResult result) {
            final T oldRecord;
            if (result.hitsCount() > 0) {
                SearchDocument doc = result.hitsArray().get(0);
                try {
                    oldRecord = recordFromDocument(doc);
                } catch (ParseException e) {
                    failed(e);
                    return;
                }
            } else {
                oldRecord = null;
            }
            try {
                updateEntityRecord(oldRecord, newRecord, session);
            } catch (Exception e) {
                failed(e);
            }
        }
    }

    private static class ResponseCallback
        extends AbstractProxySessionCallback<Void>
    {
        ResponseCallback(final ProxySession session) {
            super(session);
        }

        @Override
        public void completed(final Void v) {
            session.response(HttpStatus.SC_OK);
        }
    }
}
