package ru.yandex.search.tts.gorgophone;

import java.io.IOException;

import java.text.ParseException;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.json.writer.JsonWriter;

import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.result.SearchDocument;

public class PropertyRecordContext
    extends AbstractRecordContext<PropertyRecordContext, PropertyLuceneFields>
{
    public static final String ENTITY_ID = EntityRecordContext.ENTITY_ID;
    public static final String PROPERTY_KIND = "property-kind";
    public static final String VALUE = "value";
    public static final String CONFIDENCE = "confidence";
    public static final PropertyLuceneFields FIELD_MAP =
        new PropertyLuceneFields();

    private final String entityId;
    private final String propertyKind;
    private final String value;
    private final double confidence;

    public PropertyRecordContext(final CgiParams params)
        throws BadRequestException
    {
        super(params, FIELD_MAP);
        entityId = params.getString(ENTITY_ID);
        propertyKind = params.getString(PROPERTY_KIND);
        value = params.getString(VALUE);
        confidence = params.getDouble(CONFIDENCE);
    }

    public PropertyRecordContext(final CgiParams params, final long queueId)
        throws BadRequestException
    {
        super(params, queueId, FIELD_MAP);
        entityId = params.getString(ENTITY_ID);
        propertyKind = params.getString(PROPERTY_KIND);
        value = params.getString(VALUE);
        confidence = params.getDouble(CONFIDENCE);
    }

    public PropertyRecordContext(final SearchDocument doc)
        throws ParseException
    {
        super(doc, FIELD_MAP);
        try {
            entityId = doc.attrs().get(FIELD_MAP.entityId());
            propertyKind = doc.attrs().get(FIELD_MAP.propertyKind());
            value = doc.attrs().get(FIELD_MAP.value());
            confidence =
                Double.parseDouble(doc.attrs().get(FIELD_MAP.confidence()));
        } catch (Exception e) {
            ParseException pe = new ParseException(
                "Error parsing loaded document: " + e.getMessage(),
                0);
            pe.addSuppressed(e);
            throw pe;
        }
    }

    protected PropertyRecordContext(
        final PropertyRecordContext old,
        final PropertyRecordContext current)
    {
        super(old, current);
        entityId = current.entityId;
        propertyKind = current.propertyKind;
        value = current.value;
        confidence = current.confidence;
    }

    @Override
    public String shardingField() {
        return entityId;
    }

    @Override
    public void appendCgiParams(final QueryConstructor qc)
        throws BadRequestException
    {
        qc.append(ENTITY_ID, entityId);
        qc.append(PROPERTY_KIND, propertyKind);
        qc.append(VALUE, value);
        qc.append(CONFIDENCE, Double.toString(confidence));
    }

    @Override
    public void writeFields(
        final JsonWriter writer,
        final PropertyLuceneFields fieldMap)
        throws IOException
    {
        writer.key(fieldMap.entityId());
        writer.value(entityId);
        writer.key(fieldMap.propertyKind());
        writer.value(propertyKind);
        writer.key(fieldMap.value());
        writer.value(value);
        writer.key(fieldMap.confidence());
        writer.value(confidence);
    }

    @Override
    public PropertyRecordContext adjust(final PropertyRecordContext old) {
        return new PropertyRecordContext(old, this);
    }

    @Override
    public String id() {
        return entityId + '#' + propertyKind;
    }

    public String entityId() {
        return entityId;
    }

    public String propertyKind() {
        return propertyKind;
    }

    public String value() {
        return value;
    }

    public double confidence() {
        return confidence;
    }
}
