package ru.yandex.search.tts.gorgophone;

import java.io.IOException;

import java.text.ParseException;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.json.writer.JsonWriter;

import ru.yandex.parser.uri.CgiParams;
import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.result.SearchDocument;

public class RelationshipRecordContext
    extends AbstractRecordContext<
        RelationshipRecordContext, RelationshipLuceneFields>
{
    public static final RelationshipLuceneFields FIELD_MAP =
        new RelationshipLuceneFields();
    public static final String RELATIONSHIP_KIND = "relationship-kind";
    public static final String ENTITY_ID1 = "entity-id1";
    public static final String ENTITY_ID2 = "entity-id2";
    public static final String CONFIDENCE = "confidence";

    private final String relationshipKind;
    private final String entityId1;
    private final String entityId2;
    private final double confidence;

    public RelationshipRecordContext(final CgiParams params)
        throws BadRequestException
    {
        super(params, FIELD_MAP);
        relationshipKind = params.getString(RELATIONSHIP_KIND);
        entityId1 = params.getString(ENTITY_ID1);
        entityId2 = params.getString(ENTITY_ID2);
        confidence = params.getDouble(CONFIDENCE);
    }

    public RelationshipRecordContext(final CgiParams params, final long queueId)
        throws BadRequestException
    {
        super(params, queueId, FIELD_MAP);
        relationshipKind = params.getString(RELATIONSHIP_KIND);
        entityId1 = params.getString(ENTITY_ID1);
        entityId2 = params.getString(ENTITY_ID1);
        confidence = params.getDouble(CONFIDENCE);
    }

    public RelationshipRecordContext(final SearchDocument doc)
        throws ParseException
    {
        super(doc, FIELD_MAP);
        try {
            entityId1 = doc.attrs().get(FIELD_MAP.entityId1());
            entityId2 = doc.attrs().get(FIELD_MAP.entityId2());
            relationshipKind = doc.attrs().get(FIELD_MAP.relationshipKind());
            confidence =
                Double.parseDouble(doc.attrs().get(FIELD_MAP.confidence()));
        } catch (Exception e) {
            ParseException pe = new ParseException(
                "Error parsing loaded document: " + e.getMessage(),
                0);
            pe.addSuppressed(e);
            throw pe;
        }
    }

    protected RelationshipRecordContext(
        final RelationshipRecordContext old,
        final RelationshipRecordContext current)
    {
        super(old, current);
        relationshipKind = current.relationshipKind;
        entityId1 = current.entityId1;
        entityId2 = current.entityId2;
        confidence = current.confidence;
    }

    @Override
    public String shardingField() {
        return entityId1;
    }

    @Override
    public void appendCgiParams(final QueryConstructor qc)
        throws BadRequestException
    {
        qc.append(RELATIONSHIP_KIND, relationshipKind);
        qc.append(ENTITY_ID1, entityId1);
        qc.append(ENTITY_ID2, entityId2);
        qc.append(CONFIDENCE, Double.toString(confidence));
    }

    @Override
    public void writeFields(
        final JsonWriter writer,
        final RelationshipLuceneFields fieldMap)
        throws IOException
    {
        writer.key(fieldMap.relationshipKind());
        writer.value(relationshipKind);
        writer.key(fieldMap.entityId1());
        writer.value(entityId1);
        writer.key(fieldMap.entityId2());
        writer.value(entityId2);
        writer.key(fieldMap.confidence());
        writer.value(confidence);
    }

    @Override
    public RelationshipRecordContext adjust(
        final RelationshipRecordContext old)
    {
        return new RelationshipRecordContext(old, this);
    }

    @Override
    public String id() {
        return entityId1 + '#' + entityId2 + '#' + relationshipKind;
    }

    public String entityId1() {
        return entityId1;
    }

    public String entityId2() {
        return entityId2;
    }

    public String relationshipKind() {
        return relationshipKind;
    }

    public double confidence() {
        return confidence;
    }
}
