package ru.yandex.major;

import java.io.IOException;

import java.text.ParseException;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Set;

import java.util.stream.Collectors;

import org.apache.http.HttpHost;
import org.apache.http.message.BasicHttpRequest;

import ru.yandex.collection.Pattern;

import ru.yandex.concurrent.TimeFrameQueue;

import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;

import ru.yandex.major.config.ImmutableMajorConfig;

import ru.yandex.parser.searchmap.SearchMap;
import ru.yandex.parser.searchmap.SearchMapHost;
import ru.yandex.parser.searchmap.SearchMapShard;
import ru.yandex.parser.searchmap.User;

import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.proxy.SearchProxy;

import ru.yandex.stater.IntegralSumAggregatorFactory;
import ru.yandex.stater.NamedStatsAggregatorFactory;
import ru.yandex.stater.PassiveStaterAdapter;

public class Major extends SearchProxy<ImmutableMajorConfig> {
    private final HttpHost currentHost;
    private final AsyncClient proxyClient;
    private final AsyncClient producerClient;
    private final Storage storage;

    private final TimeFrameQueue<Long> yuidsUpdates;
    private final TimeFrameQueue<Long> onlineTransfer;
    private final Map<Integer, Set<HttpHost>> majorMap;

    public Major(final ImmutableMajorConfig config) throws IOException {
        super(config);

        HttpHost currentHost = null;
        String hostname = System.getProperty("HOSTNAME");
        for (HttpHost host: searchMap.indexerHosts()) {
            if (host.getHostName().equalsIgnoreCase(hostname)) {
                currentHost = host;
                break;
            }
        }

        if (currentHost == null) {
            throw new IOException(
                "Current hostname " + hostname
                    + " was not found in searchmap " + searchMap.toString());
        }

        this.currentHost = currentHost;

        producerClient =
            client("ProducerClient", config.producerConfig())
                .adjustStater(
                    config.staters(),
                    new RequestInfo(
                        new BasicHttpRequest(
                            RequestHandlerMapper.GET,
                            "/producer/client")));
        proxyClient =
            client("ProxyClient", config.proxyConfig())
                .adjustStater(
                    config.staters(),
                    new RequestInfo(
                        new BasicHttpRequest(
                            RequestHandlerMapper.GET,
                            "/proxy/client")));

        StorageUpdatesConsumer updatesConsumer =
            new BasicStorageUpdatesConsumer(this);
        if (config.storageConfig().dumpPath() != null) {
            this.storage =
                new MemoryDumpingStorage(
                    this,
                    updatesConsumer,
                    config.storageConfig());
            logger().info("Creating dumping storage");
        } else {
            this.storage =
                new MemoryStorage(
                    this,
                    updatesConsumer,
                    config.storageConfig());
            logger().info("Creating simple storage");
        }

        registerStater(this.storage);

        yuidsUpdates = new TimeFrameQueue<>(config.metricsTimeFrame());
        onlineTransfer = new TimeFrameQueue<>(config.metricsTimeFrame());

        registerStater(
            new PassiveStaterAdapter<>(
                yuidsUpdates,
                new NamedStatsAggregatorFactory<>(
                     "yuids-updates_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        registerStater(
            new PassiveStaterAdapter<>(
                onlineTransfer,
                new NamedStatsAggregatorFactory<>(
                    "online-transfers_ammm",
                    IntegralSumAggregatorFactory.INSTANCE)));

        this.majorMap = loadMajorSearchmap();

        register(
            new Pattern<>("/update", false),
            new UserUpdateHandler(this, storage));
        register(
            new Pattern<>("/get", false),
            new UserInfoHandler(this, storage));
        register(
            new Pattern<>("/online", false),
            new UserInfoHandler(this, storage));
        register(
            new Pattern<>("/forceYuidsUpdate", false),
            new YuidsUpdateHandler(this, updatesConsumer));
    }

    protected Map<Integer, Set<HttpHost>> loadMajorSearchmap()
        throws IOException
    {
        try {
            SearchMap searchmap = this.config().searchMapConfig().build();
            Map<Integer, Set<HttpHost>> shards = new LinkedHashMap<>();
            for (int i = 0; i < SearchMap.SHARDS_COUNT; i++) {
                SearchMapShard shard =
                    searchmap.hosts(new User("major", new LongPrefix(i)));
                shards.put(
                    i,
                    shard.stream()
                        .map(SearchMapHost::indexerHost)
                        .collect(Collectors.toSet()));
            }

            return Collections.unmodifiableMap(shards);
        } catch (IOException | ParseException e) {
            throw new IOException("Failed to read searchmap", e);
        }
    }

    @Override
    public void start() throws IOException {
        super.start();

        storage.start();
    }

    @Override
    public void close() throws IOException {
        super.close();

        storage.close();
    }

    public HttpHost currentHost() {
        return currentHost;
    }

    public AsyncClient msearchProxyClient() {
        return proxyClient;
    }

    public HttpHost msearchProxyHost() {
        return config.proxyConfig().host();
    }

    public AsyncClient producerClient() {
        return producerClient;
    }

    public HttpHost producerHost() {
        return config.producerConfig().host();
    }

    public TimeFrameQueue<Long> yuidsUpdates() {
        return yuidsUpdates;
    }

    public TimeFrameQueue<Long> onlineTransfer() {
        return onlineTransfer;
    }

    public Map<Integer, Set<HttpHost>> majorMap() {
        return majorMap;
    }

    public Set<HttpHost> userHosts(final long uid) {
        return majorMap.getOrDefault(
            (int) (uid % SearchMap.SHARDS_COUNT),
            Collections.emptySet());
    }

    public boolean userOnCurrentHost(final long uid) {
        return userHosts(uid).contains(currentHost);
    }
}
