package ru.yandex.major;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStreamWriter;

import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;

import java.util.Arrays;
import java.util.LinkedHashSet;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;

import ru.yandex.major.config.ImmutableStorageConfig;
import ru.yandex.parser.searchmap.User;
import ru.yandex.search.prefix.LongPrefix;

public class MemoryDumpingStorage
    extends MemoryStorage
    implements Runnable
{
    private static final int DUMP_CHECK_PERIOD = 200;

    protected volatile boolean stop = false;
    private final Thread dumpingThread;

    public MemoryDumpingStorage(
        final Major major,
        final StorageUpdatesConsumer updatesConsumer,
        final ImmutableStorageConfig config)
    {
        super(major, updatesConsumer, config);

        if (config.dumpPath() != null) {
            load();
        }

        this.dumpingThread = new Thread(this, "StorageDumper");
    }

    public void start() {
        dumpingThread.start();
    }

    private void load() {
        try {
            final int parts = 3;
            final AtomicLong loaded = new AtomicLong();
            Files.lines(new File(config.dumpPath()).toPath())
                .forEach(l -> {
                    String[] split = l.split("\\t");
                    if (split.length == parts) {
                        Long key = Long.parseLong(split[0]);
                        User user = new User("major", new LongPrefix(key));
                        if (major.searchMap().indexerHosts(user).contains(
                            major.currentHost()))
                        {
                            Long ts = Long.parseLong(split[1]);
                            Set<String> yuids =
                                new LinkedHashSet<>(
                                    Arrays.asList(
                                        split[2].split(",")));
                            cache.put(key, new Item(ts, yuids));
                            loaded.incrementAndGet();
                        }
                    }
                });
            logger.info("Loaded from disk " + loaded.get());
        } catch (Exception e) {
            logger.log(Level.WARNING, "Failed to load cache", e);
        }
    }

    private boolean dump() {
        File tmpFile = new File(config.dumpPath() + ".dumping");
        File dmpFile = new File(config.dumpPath());

        logger.info("Cache dump start");

        try {
            Files.delete(tmpFile.toPath());
        } catch (Exception nfe) {
            logger.info("Previous dump do not exists, continue");
        }

        long records = 0;
        try (BufferedWriter bw =
                 new BufferedWriter(
                     new OutputStreamWriter(
                         new FileOutputStream(tmpFile),
                         StandardCharsets.UTF_8)))
        {
            final char sep = '\t';
            for (Map.Entry<Long, Item> item: cache.entrySet()) {
                StringBuilder sb = new StringBuilder();
                sb.append(item.getKey());
                sb.append(sep);
                sb.append(item.getValue().ts());
                sb.append(sep);

                if (item.getValue().yuids().size() > 0) {
                    for (String yuid: item.getValue().yuids()) {
                        sb.append(yuid);
                        sb.append(',');
                    }
                }

                bw.write(sb.toString());
                bw.newLine();
                records += 1;
            }

            logger.info("Dumped records " + records);
        } catch (IOException e) {
            logger.log(Level.WARNING, "Cache dump failed", e);
            return false;
        }

        boolean dumped = false;
        try {
            Files.move(
                tmpFile.toPath(),
                dmpFile.toPath(),
                StandardCopyOption.REPLACE_EXISTING);

            dumped = true;
            logger.info("cache dump finished");
        } catch (IOException | SecurityException e) {
            logger.log(Level.WARNING, "Failed to move cache dump", e);
        }

        return dumped;
    }

    @Override
    public void run() {
        long lastDumped = 0;
        try {
            while (!stop) {
                Thread.sleep(DUMP_CHECK_PERIOD);

                if (stop || System.currentTimeMillis()
                    > lastDumped + config.dumpPeriod())
                {
                    dump();
                    lastDumped = System.currentTimeMillis();
                }
            }
        } catch (InterruptedException ie) {
            logger.log(Level.WARNING, "Storage dump thread interrupted", ie);
        }

        logger.info("Dumping thread stopped");
    }

    @Override
    public void close() throws IOException {
        stop = true;
        dumpingThread.interrupt();
        final int attempts = 3;
        for (int i = 0; i < attempts; i++) {
            if (dump()) {
                break;
            }
        }
    }
}
