package ru.yandex.major;

import com.googlecode.concurrentlinkedhashmap.ConcurrentLinkedHashMap;

import java.io.IOException;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.major.config.ImmutableStorageConfig;

import ru.yandex.stater.StatsConsumer;

public class MemoryStorage implements Storage {
    protected final PrefixedLogger logger;
    protected final ConcurrentLinkedHashMap<Long, Item> cache;
    protected final ImmutableStorageConfig config;
    protected final Major major;
    protected final StorageUpdatesConsumer updatesConsumer;

    public MemoryStorage(
        final Major major,
        final StorageUpdatesConsumer updatesConsumer,
        final ImmutableStorageConfig config)
    {
        this.cache = new ConcurrentLinkedHashMap.Builder<Long, Item>()
            .initialCapacity(config.initialStorageSize())
            .maximumWeightedCapacity(config.maximumCapacity())
            .concurrencyLevel(Runtime.getRuntime().availableProcessors())
            .build();
        this.config = config;
        this.major = major;
        this.updatesConsumer = updatesConsumer;
        this.logger = major.logger().addPrefix("Storage");
    }

    @Override
    public void start() {
    }

    @Override
    public void close() throws IOException {
    }

    @Override
    public PutStatus putQuiet(
        final long uid,
        final long ts,
        final Collection<String> yuids)
    {
        return this.put(EmptyStorageUpdatesConsumer.INSTANCE, uid, ts, yuids);
    }

    @Override
    public PutStatus put(
        final long uid,
        final long ts,
        final Collection<String> yuids)
    {
        return this.put(updatesConsumer, uid, ts, yuids);
    }

    // CSOFF: ParameterNumber
    protected PutStatus put(
        final StorageUpdatesConsumer consumer,
        final long uid,
        final long ts,
        final Collection<String> yuids)
    {
        PutStatus status = PutStatus.CREATED;
        Item item = new Item(ts, new HashSet<>(yuids));
        Item previous = cache.putIfAbsent(uid, item);
        boolean yuidsChanged = false;
        if (previous != null) {
            while (true) {
                if (previous != null) {
                    Set<String> yuidsSet = new HashSet<>(previous.yuids());
                    yuidsChanged |= yuidsSet.addAll(yuids);

                    item = new Item(
                        Math.max(item.ts(), previous.ts()),
                        yuidsSet);
                    if (cache.replace(uid, previous, item)) {
                        if (yuidsChanged) {
                            status = PutStatus.UPDATED;
                        } else {
                            status = PutStatus.NOT_CHANGED;
                        }

                        break;
                    }

                    previous = cache.get(uid);
                } else {
                    previous = cache.putIfAbsent(uid, item);
                    if (previous == null) {
                        status = PutStatus.CREATED;

                        break;
                    }
                }
            }
        } else {
            yuidsChanged = yuids.size() > 0;
        }

        long cTime = System.currentTimeMillis();
        if (online(item, cTime)) {
            if (previous == null || !online(previous, cTime)) {
                StringBuilder logSb = new StringBuilder("User become online");
                logSb.append(uid);
                if (previous != null) {
                    logSb.append(" before ");
                    logSb.append(previous.ts());
                }

                logSb.append(" become ");
                logSb.append(item.ts());
                logger.info(logSb.toString());

                consumer.online(uid, ts, cTime);
            }
        }

        if (yuidsChanged) {
            consumer.yuids(uid, item.yuids());
        }

        return status;
    }
    // CSON: ParameterNumber

    private boolean online(final Item item, final long curTime) {
        if (item == null) {
            return false;
        }

        return curTime < item.ts() + config.storageExpireTime();
    }

    private boolean online(final Item item) {
        return online(item, System.currentTimeMillis());
    }

    @Override
    public boolean online(final long uid) {
        Item item = cache.get(uid);
        if (item == null) {
            return false;
        }

        return online(item);
    }

    @Override
    public long lastSeen(final long uid) {
        Item item = cache.get(uid);
        if (item == null) {
            return -1;
        }

        return item.ts();
    }

    @Override
    public Set<String> yuids(final long uid) {
        Item item = cache.get(uid);
        if (item != null) {
            return item.yuids();
        }

        return Collections.emptySet();
    }

    @Override
    public <E extends Exception> void stats(
        final StatsConsumer<? extends E> statsConsumer)
        throws E
    {
        statsConsumer.stat("storage-cache-capacity_ammm", cache.capacity());
        statsConsumer.stat("storage-cache-size_ammm", cache.size());
        statsConsumer.stat(
            "storage-capacity-left_ammv",
            cache.capacity() - cache.size());
    }
}
