package ru.yandex.tools.checkstyle.projects;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;

import java.nio.charset.StandardCharsets;
import java.nio.file.FileVisitResult;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.attribute.BasicFileAttributes;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public class Main {
    private static final Pattern PACKAGE = Pattern.compile("^package ");
    private static final Pattern SEMICOLON = Pattern.compile(";$");
    private static final Pattern IMPORT = Pattern.compile("^import ");
    private static final Pattern CLASS = Pattern.compile("[.][A-Z].*$");
    private static final String MAIN = "main";
    private static final String TEST = "test";
    private static final String BENCHMARK = "benchmark";
    private static final String FOR_PROJECT = "For project '";

    private final Map<String, List<String>> props = new HashMap<>();
    private final Map<String, Set<String>> projectPackages = new HashMap<>();
    private boolean failed = false;

    public Main() throws IOException {
        Properties properties = new Properties();
        try (InputStream in = new FileInputStream("projects.mk")) {
            properties.load(in);
        }
        for (Map.Entry<?, ?> entry: properties.entrySet()) {
            String key = entry.getKey().toString().trim();
            String value = entry.getValue().toString();
            if (!value.isEmpty() && value.charAt(0) == '=') {
                value = value.substring(1);
            }
            String[] strs = value.split(" ");
            List<String> values = new ArrayList<>(strs.length);
            for (String str: strs) {
                value = str.trim();
                if (!value.isEmpty()) {
                    values.add(value);
                }
            }
            for (int i = 1; i < values.size(); ++i) {
                if (values.get(i - 1).compareTo(values.get(i)) >= 0
                    && !values.get(i - 1).equals("$(sort"))
                {
                    failed = true;
                    System.err.println(
                        "For key '" + key
                        + "' values order is incorrect at position #" + i
                        + ": " + values.get(i - 1) + " >= " + values.get(i));
                }
            }
            props.put(key, values);
        }
        for (String project: projects()) {
            projectPackages.put(project, projectPackages(project));
        }
    }

    public List<String> projects() {
        return props.get("projects");
    }

    private List<String> projectDeps(final String project, final String type) {
        List<String> deps = props.get(project + '-' + type + "-deps");
        if (deps == null) {
            deps = Collections.emptyList();
        }
        return deps;
    }

    private void populateDeps(final String project, final Set<String> deps) {
        List<String> projectDeps = projectDeps(project, MAIN);
        deps.addAll(projectDeps);
        for (String dep: projectDeps) {
            populateDeps(dep, deps);
        }
    }

    private static Set<String> projectPackages(final String project)
        throws IOException
    {
        return projectLines(project, MAIN, PACKAGE, SEMICOLON);
    }

    private static Set<String> projectImports(
        final String project,
        final String type)
        throws IOException
    {
        return projectLines(project, type, IMPORT, CLASS);
    }

    // CSOFF: ParameterNumber
    private static Set<String> projectLines(
        final String project,
        final String type,
        final Pattern firstErasePattern,
        final Pattern secondErasePattern)
        throws IOException
    {
        Path path = Paths.get("src/" + project + '/' + type + "/java");
        Set<String> lines = new HashSet<>();
        if (Files.exists(path)) {
            Files.walkFileTree(
                path,
                new PrefixedLinesLoader(
                    lines,
                    firstErasePattern,
                    secondErasePattern));
        }
        return lines;
    }
    // CSON: ParameterNumber

    private static class PrefixedLinesLoader extends SimpleFileVisitor<Path> {
        private final Set<String> lines;
        private final Pattern firstErasePattern;
        private final Pattern secondErasePattern;

        PrefixedLinesLoader(
            final Set<String> lines,
            final Pattern firstErasePattern,
            final Pattern secondErasePattern)
        {
            this.lines = lines;
            this.firstErasePattern = firstErasePattern;
            this.secondErasePattern = secondErasePattern;
        }

        @Override
        public FileVisitResult visitFile(
            final Path file,
            final BasicFileAttributes attrs)
            throws IOException
        {
            if (file.toString().endsWith(".java")) {
                for (String line
                    : Files.readAllLines(file, StandardCharsets.UTF_8))
                {
                    String trimmed = line.trim();
                    Matcher matcher = firstErasePattern.matcher(trimmed);
                    if (matcher.find()) {
                        lines.add(
                            secondErasePattern.matcher(matcher.replaceAll(""))
                                .replaceAll("")
                                .trim());
                    }
                }
            }
            return FileVisitResult.CONTINUE;
        }
    }

    public Set<String> checkDeps(
        final String project,
        final String type,
        final Set<String> parentDeps)
        throws IOException
    {
        List<String> projectDeps = projectDeps(project, type);
        Set<String> deps = new HashSet<>(parentDeps);
        for (String dep: projectDeps) {
            populateDeps(dep, deps);
        }
        Set<String> projectPackages = this.projectPackages.get(project);
        Set<String> imports = projectImports(project, type);
        for (String dep: projectDeps) {
            if (deps.contains(dep)) {
                failed = true;
                System.err.println(
                    FOR_PROJECT + project
                    + "' unnecessary " + type
                    + " dependency detected: " + dep);
            }
            Set<String> packagesInDependency = this.projectPackages.get(dep);
            if (packagesInDependency == null) {
                failed = true;
                System.err.println(
                    "Typo in dependencies list for project " + project
                    + '(' + type + "). Unknown dependency " + dep);
            } else if (!packagesInDependency.isEmpty()
                && !projectPackages.isEmpty())
            {
                boolean intersects = false;
                for (String packageName: packagesInDependency) {
                    if (imports.contains(packageName)) {
                        intersects = true;
                        break;
                    }
                }
                if (!intersects) {
                    failed = true;
                    System.err.println(
                        FOR_PROJECT + project
                        + '\'' + ' ' + type
                        + " no imported packages found from: " + dep);
                }
            }
        }
        return deps;
    }

    public static void main(final String... args) throws IOException {
        Main main = new Main();
        for (String project: main.projects()) {
            main.checkDeps(
                project,
                BENCHMARK,
                main.checkDeps(
                    project,
                    TEST,
                    main.checkDeps(project, MAIN, Collections.emptySet())));
        }
        if (main.failed) {
            System.exit(1);
        }
    }
}

