package ru.yandex.mail.search.staff.consumer;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.http.util.BadResponseException;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.mail.search.staff.Person;
import ru.yandex.mail.search.staff.Persons;

public class StaffCallback implements FutureCallback<Persons> {
    private final PrefixedLogger logger;
    private final StaffUpdateTaskCallback callback;
    private final StaffConsumer consumer;

    public StaffCallback(
        final StaffConsumer consumer,
        final StaffUpdateTaskCallback callback)
    {
        this.callback = callback;
        this.consumer = consumer;
        this.logger = consumer.logger();
    }

    @Override
    public void completed(final Persons result) {
        LinkedHashMap<Long, List<Person>> grouped =
            new LinkedHashMap<>(result.people().size());

        for (Person person: result.people()) {
            grouped.computeIfAbsent(
                person.messageId(),
                (k) -> new ArrayList<>())
                .add(person);
        }
        Iterator<Map.Entry<Long, List<Person>>> iterator =
            grouped.entrySet().iterator();

        long lastPosition = -1;
        if (result.people().size() > 0) {
            lastPosition =
                result.people().get(result.people().size() - 1).messageId();
        }

        try {
            for (int i = 0; i < grouped.size(); i++) {
                Map.Entry<Long, List<Person>> entry = iterator.next();
                //on last record in batch we do not save position in producer
                //because in the next batch could be same position
                long position = entry.getKey();
                if (i == grouped.size() - 1) {
                    position = -1;
                }

                consumer.consume(
                    callback.lockId(),
                    position,
                    entry.getValue());
            }
        } catch (BadResponseException e) {
            callback.failed(e);
            return;
        }

        logger.fine(
            "Consumed records: " + result.people().size()
                + " message id " + lastPosition);
        if (result.next() != null && !result.next().isEmpty()) {
            callback.updateMessageId(lastPosition);
            consumer.load(result.next(), callback);
        } else {
            logger.info(
                "Consume completed, position is " + lastPosition);
            callback.completed(null);
        }
    }

    @Override
    public void failed(final Exception e) {
        logger.log(
            Level.WARNING,
            "Failed to load data from staff",
            e);

        if (e instanceof BadResponseException) {
            BadResponseException bre = (BadResponseException) e;
            if (bre.response() != null
                && bre.response().contains(
                    "Overflow sort stage buffered data usage"))
            {
                consumer.logger().info(
                    "Overflow in db due to sort, "
                        + "trying fetch single message");

                consumer.loadMessage(
                    new SingleMessageUpdateTaskCallback(consumer, callback),
                    callback.startMessageId());
                return;
            }
        }

        callback.failed(e);
    }

    @Override
    public void cancelled() {
        logger.warning("Staff request cancelled");
        callback.cancelled();
    }
}
