package ru.yandex.mail.search.staff.consumer;

import java.io.IOException;
import java.net.InetAddress;
import java.net.URISyntaxException;
import java.net.UnknownHostException;

import java.util.List;
import java.util.TimerTask;

import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;

import java.util.logging.Level;

import org.apache.http.HttpException;
import org.apache.http.HttpResponse;
import org.apache.http.HttpStatus;

import org.apache.http.client.methods.HttpGet;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.BadResponseException;
import ru.yandex.http.util.CharsetUtils;
import ru.yandex.http.util.EmptyFutureCallback;
import ru.yandex.http.util.YandexHeaders;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.StatusCheckAsyncResponseConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.io.StringBuilderWriter;

import ru.yandex.json.writer.JsonType;
import ru.yandex.json.writer.JsonWriter;

import ru.yandex.logger.PrefixedLogger;

import ru.yandex.mail.search.staff.Person;
import ru.yandex.mail.search.staff.config.ImmutableStaffConsumerConfig;

import ru.yandex.parser.uri.QueryConstructor;

public class StaffConsumer extends TimerTask {
    public static final String EQUALS_OPERATION = "==";
    public static final String GREATER_OR_EQUALS_OPERATION = ">=";
    public static final String GREATER_OPERATION = ">";

    private static final StatusCheckAsyncResponseConsumerFactory<HttpResponse>
        PRODUCER_CONSUMER_FACTORY =
        new StatusCheckAsyncResponseConsumerFactory<>(
            x -> x == HttpStatus.SC_OK || x == HttpStatus.SC_CONFLICT,
            BasicAsyncResponseConsumerFactory.INSTANCE);

    private static final String FIELDS
        = "name,login,official,language,uid,_meta.message_id";

    private static final String MESSAGE_ID = "_meta.message_id";
    private static final String PRODUCER_NAME = "staff-consumer";
    private static final String PRODUCER_NAME_PARAM = "&producer-name=";
    private static final long DEFAULT_PREFIX = 0L;

    private final AsyncClient staffClient;
    private final AsyncClient producerClient;

    private final String name;

    private final String lockRequest;
    private final String getPositionRequest;

    private final PrefixedLogger logger;
    private final ImmutableStaffConsumerConfig config;

    private volatile String token;

    public StaffConsumer(final StaffConsumerServer server) {
        this.logger = server.logger();
        this.config = server.config();

        String name = System.getProperty("BSCONFIG_IHOST");
        if (name == null) {
            try {
                name = InetAddress.getLocalHost().getHostAddress();
            } catch (UnknownHostException ue) {
                name = "unknown";
            }
        }

        staffClient = server.staffClient();
        producerClient = server.producerClient();

        lockRequest =
            "/_lock?service=" + config.service()
                + "&timeout=" + config.lockTimeout()
                + "&name=" + PRODUCER_NAME + "&id=";
        getPositionRequest =
            "/_producer_position?service=" + config.service()
                + PRODUCER_NAME_PARAM + PRODUCER_NAME;

        this.name = name;
    }

    public PrefixedLogger logger() {
        return logger;
    }

    @Override
    public void run() {
        if (token == null) {
            token = this.name + "@1";
        }

        boolean locked = tryLock(token);
        Long messageId = lastMessageId();

        if (locked && messageId != null) {
            WaitingStaffUpdateTaskCallback task =
                new WaitingStaffUpdateTaskCallback(logger(), token, messageId);
            logger().info("Starting staff update");

            load(task);

            WaitingStaffUpdateTaskCallback.TaskStatus status = task.get();
            if (status == WaitingStaffUpdateTaskCallback.TaskStatus.COMPLETED) {
                logger().info("Staff update finished");
            } else {
                logger().info("Staff update failed");
            }
        }
    }

    protected QueryConstructor stafRequestBase(
        final String operation,
        final Long sId)
        throws BadRequestException
    {
        QueryConstructor qc =
            new QueryConstructor(config.staffClientConfig().uri().toString());

        qc.append("_fields", FIELDS);
        qc.append(
            "_query",
            MESSAGE_ID + operation + sId);

        return qc;
    }

    protected void loadMessage(
        final StaffUpdateTaskCallback callback,
        final Long sId)
    {
        try {
            QueryConstructor qc = stafRequestBase(EQUALS_OPERATION, sId);
            load(qc.toString(), callback);
        } catch (BadRequestException bre) {
            callback.failed(bre);
        }
    }

    protected void load(
        final StaffUpdateTaskCallback callback,
        final String operation,
        final Long sId)
    {
        try {
            QueryConstructor qc = stafRequestBase(operation, sId);
            qc.append("_sort", MESSAGE_ID);

            load(qc.toString(), callback);
        } catch (BadRequestException bre) {
            callback.failed(bre);
        }
    }

    protected void load(final StaffUpdateTaskCallback callback) {
        load(callback, GREATER_OR_EQUALS_OPERATION, callback.startMessageId());
    }

    protected void load(
        final String uri,
        final StaffUpdateTaskCallback callback)
    {
        logger().info("Processing staff request " + uri);

        try {
            staffClient.execute(
                new StaffRequestProducerSupplier(uri, config.token()),
                PersonsConsumerFactory.OK,
                staffClient.httpClientContextGenerator(),
                new StaffCallback(this, callback));
        } catch (URISyntaxException e) {
            callback.failed(e);
        }
    }

    protected Exception consume(
        final String lockId,
        final long messageId,
        final List<Person> people)
        throws BadResponseException
    {
        StringBuilder uri = new StringBuilder("/update?");
        uri.append("&staffloader=");
        uri.append(name);
        uri.append("&prefix=");
        uri.append(DEFAULT_PREFIX);
        uri.append("&service=");
        uri.append(config.service());
        uri.append("&message-id=");
        uri.append(messageId);
        uri.append("&batch-size=");
        uri.append(people.size());

        StringBuilderWriter sbw = new StringBuilderWriter();
        try (JsonWriter writer = JsonType.HUMAN_READABLE.create(sbw)) {
            writer.startObject();
            writer.key("prefix");
            writer.value(DEFAULT_PREFIX);
            writer.key("AddIfNotExists");
            writer.value(true);
            writer.key("docs");
            writer.startArray();
            for (Person person: people) {
                writer.startObject();
                person.writeValue(writer);
                writer.endObject();
            }
            writer.endArray();
            writer.endObject();
        } catch (IOException ioe) {
            logger().log(Level.WARNING, "Failed to build index request", ioe);
            return ioe;
        }

        String indexBody = sbw.toString();
        BasicAsyncRequestProducerGenerator request =
            new BasicAsyncRequestProducerGenerator(
                uri.toString(),
                indexBody);
        request.addHeader(YandexHeaders.SERVICE, config.service());
        request.addHeader(YandexHeaders.PRODUCER_NAME, PRODUCER_NAME);

        if (messageId >= 0) {
            request.addHeader(
                YandexHeaders.PRODUCER_POSITION,
                String.valueOf(messageId));
        }

        logger().info("Index request " + request.toString());

        Future<HttpResponse> future =
            producerClient.execute(
                config.producerConfig().host(),
                request,
                PRODUCER_CONSUMER_FACTORY,
                EmptyFutureCallback.INSTANCE);
        Exception result = null;
        try {
            HttpResponse response = future.get();
            int status = response.getStatusLine().getStatusCode();
            if (status == HttpStatus.SC_CONFLICT) {
                logger.warning(
                    "Skipping, messageId that already indexed "
                        + messageId);
            }
        } catch (NumberFormatException
            | InterruptedException
            | ExecutionException e)
        {
            logger().log(
                Level.WARNING,
                "Producer request failed " + request.toString(),
                e);
            result = e;
        }

        return result;
    }

    protected boolean tryLock(final String lockId) {
        HttpGet request = new HttpGet(lockRequest);
        Future<HttpResponse> future =
            producerClient.execute(
                config.producerConfig().host(),
                new BasicAsyncRequestProducerGenerator(lockRequest + lockId),
                EmptyFutureCallback.INSTANCE);
        try {
            HttpResponse response = future.get();
            int status = response.getStatusLine().getStatusCode();
            if (status == HttpStatus.SC_FORBIDDEN) {
                logger().fine(
                    "Failed to obtain lock, current is "
                        + CharsetUtils.toString(response.getEntity()));
            } else if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(request, response);
            } else {
                return true;
            }
        } catch (IOException
            | HttpException
            | InterruptedException
            | ExecutionException e)
        {
            logger().log(Level.WARNING, "Failed to get lock", e);
        }

        return false;
    }

    protected Long lastMessageId() {
        Future<HttpResponse> future =
            producerClient.execute(
                config.producerConfig().host(),
                new BasicAsyncRequestProducerGenerator(getPositionRequest),
                EmptyFutureCallback.INSTANCE);
        try {
            HttpResponse response = future.get();
            int status = response.getStatusLine().getStatusCode();
            if (status != HttpStatus.SC_OK) {
                throw new BadResponseException(
                    new HttpGet(getPositionRequest),
                    response);
            }
            String body = CharsetUtils.toString(response.getEntity());

            long messageId = Long.parseLong(body);
            logger().info(
                "Last message id " + messageId);
            return messageId;
        } catch (IOException
            | HttpException
            | NumberFormatException
            | InterruptedException
            | ExecutionException e)
        {
            logger().log(Level.WARNING, "Failed to get position", e);
        }

        return null;
    }
}
