package ru.yandex.search.mail.xavier;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.http.Header;
import ru.yandex.blackbox.BlackboxUserinfo;
import ru.yandex.dbfields.ChangeType;
import ru.yandex.dbfields.OracleFields;
import ru.yandex.dbfields.PgFields;

import ru.yandex.http.proxy.ProxySession;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.HeadersParser;
import ru.yandex.http.util.YandexHeaders;

import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;
import ru.yandex.parser.searchmap.User;
import ru.yandex.search.prefix.LongPrefix;
import ru.yandex.search.prefix.Prefix;

public class BasicXavierContext implements XavierContext {
    private final Xavier xavier;
    private final User user;
    private final long queueId;
    private final Long operationId;
    private final Double operationDate;
    private final Long lcn;
    private final String pgshard;
    private final ChangeType changeType;
    private final List<String> mids;
    private final XavierCallback callback;
    private final ProxySession session;
    private final LongPrefix prefix;
    private final Map<?, ?> json;
    private final int retryCount;

    public BasicXavierContext(
        final Xavier xavier,
        final ProxySession session,
        final Map<?, ?> json)
        throws BadRequestException, JsonUnexpectedTokenException
    {
        this.session = session;
        this.xavier = xavier;

        String mdb = session.params().getString(OracleFields.MDB);
        prefix = new LongPrefix(ValueUtils.asLong(json.get(PgFields.UID)));

        this.user = new User(resolveService(mdb, prefix), prefix);
        this.json = json;

        int retryCount = 0;
        try {
            Header header =
                this.session.request().getFirstHeader("retry-count");
            if (header != null) {
                retryCount = Integer.parseInt(header.getValue());
            }
        } catch (NumberFormatException nfe) {
            session.logger().info("Failed to parse retry count");
        }

        this.retryCount = retryCount;

        changeType = ValueUtils.asEnum(
            ChangeType.class,
            json.get(PgFields.CHANGE_TYPE));
        List<?> changed = ValueUtils.asListOrNull(json.get(PgFields.CHANGED));
        if (changed != null) {
            this.mids = new ArrayList<>(changed.size());

            for (Object item: changed) {
                String mid = ValueUtils.asStringOrNull(
                    ValueUtils.asMap(item).get(PgFields.MID));
                if (mid != null) {
                    this.mids.add(mid);
                }
            }
        } else {
            this.mids = Collections.emptyList();
        }

        this.operationDate = session.params().getDouble("operation-date", null);

        Long queueId = session.params().getLong("zoo-queue-id", null);

        if (queueId == null) {
            this.queueId = new HeadersParser(session.request())
                .getLong(YandexHeaders.ZOO_QUEUE_ID, null);
        } else {
            this.queueId = queueId;
        }

        this.operationId =
            ValueUtils.asLongOrNull(json.get(PgFields.OPERATION_ID));
        this.lcn = ValueUtils.asLongOrNull(json.get(OracleFields.LCN));

        this.pgshard = ValueUtils.asStringOrNull(json.get(PgFields.PGSHARD));

        this.callback = new XavierCallback(session, this);
    }

    @Override
    public String shard() {
        return pgshard;
    }

    @Override
    public List<String> mids() {
        return mids;
    }

    @Override
    public User user() {
        return user;
    }

    public long queueId() {
        return queueId;
    }

    public ChangeType changeType() {
        return changeType;
    }

    @Override
    public XavierCallback callback() {
        return callback;
    }

    @Override
    public Long operationId() {
        return operationId;
    }

    @Override
    public Long lcn() {
        return lcn;
    }

    @Override
    public ProxySession session() {
        return session;
    }

    public static String resolveService(
        final String mdb,
        final Prefix prefix)
    {
        String result;
        if ("pg".equals(mdb)) {
            result = "change_log";
        } else if (prefix instanceof LongPrefix
            && BlackboxUserinfo.corp(((LongPrefix) prefix).prefix()))
        {
            result = "opqueue_corp";
        } else {
            result = "opqueue";
        }

        return result;
    }

    @Override
    public Double operationDate() {
        return operationDate;
    }

    @Override
    public LongPrefix prefix() {
        return prefix;
    }

    @Override
    public Map<?, ?> json() {
        return json;
    }

    @Override
    public Xavier xavier() {
        return xavier;
    }

    @Override
    public int retryCount() {
        return retryCount;
    }
}
