package ru.yandex.search.mail.xavier;

import com.google.common.cache.Cache;
import com.google.common.cache.CacheBuilder;

import java.io.IOException;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicIntegerArray;

import ru.yandex.json.writer.JsonWriter;

import ru.yandex.search.mail.xavier.config.ImmutableXavierConfig;

public class UnreadCountCache {
    private final Cache<Long, UserReadStat> cache;
    private final List<String> categories;
    private final Map<String, Integer> categoriesMap;

    public UnreadCountCache(final ImmutableXavierConfig config) {
        this.categories = new ArrayList<>(config.categories());
        this.categoriesMap = new LinkedHashMap<>();
        for (int i = 0; i < categories.size(); i++) {
            this.categoriesMap.put(this.categories.get(i), i);
        }

        cache = CacheBuilder.newBuilder()
            .maximumSize(config.unreadStatCacheSize())
            .concurrencyLevel(Runtime.getRuntime().availableProcessors())
            .expireAfterWrite(2, TimeUnit.HOURS)
            .build();
    }

    public void start() {
    }

    public UserReadStat get(final Long uid) {
        return cache.getIfPresent(uid);
    }

    public void remove(final Long prefix) {
        cache.invalidate(prefix);
    }

    public UserReadStat put(
        final Long uid,
        final Map<String, Integer> counters)
    {
        UserReadStat stat = new UserReadStat(counters);
        cache.put(uid, stat);
        return stat;
    }

    public class UserReadStat {
        private final AtomicIntegerArray stats;

        public UserReadStat() {
            this.stats = new AtomicIntegerArray(categories.size());
        }

        public UserReadStat(final int[] values) {
            this.stats = new AtomicIntegerArray(values);
        }

        public UserReadStat(final Map<String, Integer> values) {
            this.stats = new AtomicIntegerArray(values.size());
            for (Map.Entry<String, Integer> entry: values.entrySet()) {
                Integer index = categoriesMap.get(entry.getKey());
                if (index != null) {
                    this.stats.getAndAdd(index, entry.getValue());
                }
            }
        }

        public void inc(final String category) {
            stats.getAndAdd(categoriesMap.get(category), 1);
        }

        public void write(final JsonWriter writer) throws IOException {
            writer.key("counters");
            writer.startObject();
            for (Map.Entry<String, Integer> entry : categoriesMap.entrySet()) {
                writer.key(entry.getKey());
                writer.startObject();
                writer.key("unread");
                writer.value(stats.get(entry.getValue()));
                writer.endObject();
            }
        }

        public Map<String, Integer> asMap() {
            Map<String, Integer> result = new HashMap<>(categories.size());
            for (Map.Entry<String, Integer> entry : categoriesMap.entrySet()) {
                result.put(entry.getKey(), stats.get(entry.getValue()));
            }

            return result;
        }
    }
}
