package ru.yandex.search.mail.xavier;

import java.io.IOException;
import java.text.ParseException;
import java.util.EnumMap;

import java.util.Map;

import org.apache.http.HttpEntityEnclosingRequest;
import org.apache.http.HttpException;
import org.apache.http.HttpRequest;
import org.apache.http.HttpStatus;
import org.apache.http.message.BasicHttpRequest;

import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;

import org.apache.http.protocol.HttpContext;

import ru.yandex.client.producer.ProducerClient;

import ru.yandex.collection.Pattern;

import ru.yandex.dbfields.ChangeType;
import ru.yandex.dbfields.OracleFields;
import ru.yandex.dbfields.PgFields;

import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.HttpProxy;
import ru.yandex.http.proxy.ProxySession;

import ru.yandex.http.util.BadRequestException;

import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.http.util.request.RequestHandlerMapper;
import ru.yandex.http.util.request.RequestInfo;

import ru.yandex.json.async.consumer.JsonAsyncDomConsumer;

import ru.yandex.json.parser.JsonException;

import ru.yandex.json.xpath.ValueUtils;

import ru.yandex.parser.searchmap.SearchMap;

import ru.yandex.search.mail.xavier.config.ImmutableXavierConfig;
import ru.yandex.search.mail.xavier.store.StoreXavierHandler;
import ru.yandex.search.mail.xavier.update.UpdateXavierHandler;
import ru.yandex.search.mail.xavier.usertype.UsertypeXavierHandler;

public class Xavier
    extends HttpProxy<ImmutableXavierConfig>
    implements HttpAsyncRequestHandler<Object>
{
    private static final String FILTER_SEARCH_STAT_URI = "/filter-search";
    private static final String XIVA_NOTIFY_STAT_URI = "/xiva/notify";
    private static final String XIVA_LIST_STAT_URI = "/xiva/list";
    private static final String MSEARCH_PROXY_STAT_URI = "/msearch-proxy";

    private final Map<ChangeType, XavierHandler> handlers =
        new EnumMap<>(ChangeType.class);

    private final ProducerClient producerClient;
    private final AsyncClient proxyClient;
    private final AsyncClient filterSearchClient;
    private final AsyncClient searchClient;
    private final AsyncClient xivaNotifyClient;
    private final AsyncClient xivaListClient;
    private final String filterSearchUri;
    private final SearchMap searchMap;
    private final XavierStats stats;

    private final UnreadCountCache unreadCountCache;

    public Xavier(final ImmutableXavierConfig config) throws IOException {
        super(config);

        if (config.cacheUnreadStats()) {
            this.logger().info("Caching enabled");
            this.unreadCountCache = new UnreadCountCache(config);
        } else {
            this.unreadCountCache = null;
        }

        try {
            searchMap = config.searchMapConfig().build();
        } catch (ParseException e) {
            throw new IOException("Failed to parse searchmap", e);
        }

        this.searchClient = client("SearchClient", config.searchClientConfig());

        filterSearchClient =
            client("FilterSearch", config.filterSearchConfig())
                .adjustStater(
                    config.staters(),
                    new RequestInfo(
                        new BasicHttpRequest(
                            RequestHandlerMapper.GET,
                            FILTER_SEARCH_STAT_URI)));

        filterSearchUri =
            config.filterSearchConfig().uri().toASCIIString()
                + config.filterSearchConfig().firstCgiSeparator()
                + "order=default&full_folders_and_labels=1&uid=";

        xivaNotifyClient =
            client(
                "XivaNotifyClient",
                config.xivaConfig().notifyClientConfig())
                .adjustStater(
                    config.staters(),
                    new RequestInfo(
                        new BasicHttpRequest(
                            RequestHandlerMapper.GET,
                            XIVA_NOTIFY_STAT_URI)));

        xivaListClient =
            client(
                "XivaListClient",
                config.xivaConfig().listClientConfig())
                .adjustStater(
                    config.staters(),
                    new RequestInfo(
                        new BasicHttpRequest(
                            RequestHandlerMapper.GET,
                            XIVA_LIST_STAT_URI)));

        proxyClient =
            client("ProxyClient", config.proxyConfig())
                .adjustStater(
                    config.staters(),
                    new RequestInfo(
                        new BasicHttpRequest(
                            RequestHandlerMapper.GET,
                            MSEARCH_PROXY_STAT_URI)));

        producerClient =
            registerClient(
                "Producer",
                new ProducerClient(reactor, config.producerConfig(), searchMap),
                config.producerConfig());

        handlers.put(
            ChangeType.UPDATE,
            new UpdateXavierHandler(this));
        handlers.put(
            ChangeType.STORE,
            new StoreXavierHandler(this));
        handlers.put(
            ChangeType.USER_TYPE_UPDATE,
            new UsertypeXavierHandler(this));
        handlers.put(
            ChangeType.DELETE,
            CacheInvalidateHandler.INSTANCE);

        register(
            new Pattern<>("/notify", false),
            this,
            RequestHandlerMapper.POST);
        register(
            new Pattern<>("", true),
            new XavierEmptyRequestHandler(),
            RequestHandlerMapper.POST);

        this.stats = new XavierStats(this);
    }

    @Override
    public HttpAsyncRequestConsumer<Object> processRequest(
        final HttpRequest request,
        final HttpContext context)
        throws HttpException, IOException
    {
        if (!(request instanceof HttpEntityEnclosingRequest)) {
            throw new BadRequestException("Payload expected");
        }

        return new JsonAsyncDomConsumer(
            ((HttpEntityEnclosingRequest) request).getEntity());
    }

    private void handlePgRequest(
        final ProxySession session,
        final Map<?, ?> json)
        throws HttpException, JsonException
    {
        ChangeType changeType = ValueUtils.asEnum(
            ChangeType.class,
            json.get(PgFields.CHANGE_TYPE));

        XavierHandler handler = handlers.get(changeType);
        if (handler == null) {
            handler = XavierSkipHandler.INSTANCE;
        }

        XavierContext context = new BasicXavierContext(this, session, json);
        handler.handle(context);
    }

    @Override
    public void handle(
        final Object payload,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        ProxySession session = new BasicProxySession(this, exchange, context);
        String mdb = session.params().getString(OracleFields.MDB);

        try {
            Map<?, ?> json = ValueUtils.asMap(payload);
            if (mdb.equals("pg")) {
                handlePgRequest(session, json);
            } else {
                session.logger().warning(
                    "Unsupported oracle notify, skipping");
                session.response(HttpStatus.SC_OK);
            }
        } catch (JsonException e) {
            throw new BadRequestException(
                "Failed to parse request: " + payload,
                e);
        }
    }

    public ProducerClient producerClient() {
        return producerClient;
    }

    public AsyncClient filterSearchClient() {
        return filterSearchClient;
    }

    public AsyncClient searchClient() {
        return searchClient;
    }

    public AsyncClient proxyClient() {
        return proxyClient;
    }

    public AsyncClient xivaNotifyClient() {
        return xivaNotifyClient;
    }

    public AsyncClient xivaListClient() {
        return xivaListClient;
    }

    public String filterSearchUri() {
        return filterSearchUri;
    }

    public SearchMap searchMap() {
        return searchMap;
    }

    public XavierStats staters() {
        return stats;
    }

    public UnreadCountCache unreadCountCache() {
        return unreadCountCache;
    }
}
