package ru.yandex.search.mail.xavier.config;

import java.util.LinkedHashSet;
import java.util.Set;

import ru.yandex.client.producer.ProducerClientConfig;
import ru.yandex.client.producer.ProducerClientConfigBuilder;

import ru.yandex.http.config.FilterSearchConfig;
import ru.yandex.http.config.FilterSearchConfigBuilder;
import ru.yandex.http.config.HttpHostConfig;
import ru.yandex.http.config.HttpHostConfigBuilder;
import ru.yandex.http.config.HttpTargetConfig;
import ru.yandex.http.config.HttpTargetConfigBuilder;

import ru.yandex.http.util.server.AbstractHttpProxyConfigBuilder;

import ru.yandex.parser.config.ConfigException;
import ru.yandex.parser.config.IniConfig;

import ru.yandex.parser.searchmap.SearchMapConfig;
import ru.yandex.parser.searchmap.SearchMapConfigBuilder;

import ru.yandex.parser.string.CollectionParser;

import ru.yandex.search.mail.xavier.XivaConfigBuilder;

public abstract class AbstractXavierConfigBuilder<T extends
    AbstractXavierConfigBuilder<T>>
    extends AbstractHttpProxyConfigBuilder<T>
    implements XavierConfig
{
    private FilterSearchConfigBuilder filterSearchConfig;
    private XivaConfigBuilder xivaConfig;
    private HttpHostConfigBuilder proxyConfig;
    private HttpTargetConfigBuilder searchClientConfig;
    private ProducerClientConfigBuilder producerClientConfig;
    private SearchMapConfigBuilder searchMapConfig;
    private String pgQueue;
    private String indexUri;
    private int usertypeBatchSize;
    private int hoursSkipStaleRecords;
    private Set<String> categories;
    private int unreadStatCacheSize;
    private boolean cacheUnreadStats;

    protected AbstractXavierConfigBuilder(final XavierConfig config) {
        super(config);

        filterSearchConfig(config.filterSearchConfig());
        xivaConfig(config.xivaConfig());
        proxyConfig(config.proxyConfig());
        searchClientConfig(config.searchClientConfig());
        producerConfig(config.producerConfig());
        pgQueue(config.pgQueue());
        indexUri(config.indexUri());
        usertypeBatchSize(config.usertypeBatchSize());
        categories(config.categories());
        searchMapConfig(config.searchMapConfig());
        hoursSkipStaleRecords(config.hoursSkipStaleRecords());
        unreadStatCacheSize(config.unreadStatCacheSize());
        cacheUnreadStats(config.cacheUnreadStats());
    }

    protected AbstractXavierConfigBuilder(
        final IniConfig config,
        final XavierConfig defaults)
        throws ConfigException
    {
        super(config, defaults);

        this.filterSearchConfig =
            new FilterSearchConfigBuilder(
                config.section("filter-search"),
                defaults.filterSearchConfig());

        this.xivaConfig =
            new XivaConfigBuilder(
                config.section("xiva"),
                defaults.xivaConfig());

        this.proxyConfig =
            new HttpHostConfigBuilder(
                config.section("proxy"),
                defaults.proxyConfig());

        this.searchClientConfig =
            new HttpTargetConfigBuilder(
                config.section("search"),
                defaults.searchClientConfig());

        this.producerClientConfig =
            new ProducerClientConfigBuilder(
                config.section("producer"),
                defaults.producerConfig());

        this.pgQueue = config.getString("pgQueue", defaults.pgQueue());
        this.indexUri = config.getString("index-uri", defaults.indexUri());
        this.usertypeBatchSize =
            config.getInt("usertype-batch-size", defaults.usertypeBatchSize());
        this.hoursSkipStaleRecords =
            config.getInt(
                "skip-records-hours",
                defaults.hoursSkipStaleRecords());

        this.categories =
            config.get(
                "categories",
                defaults.categories(),
                new CollectionParser<>(String::trim, LinkedHashSet::new));

        this.searchMapConfig =
            new SearchMapConfigBuilder(
                config.section("searchmap"),
                defaults.searchMapConfig());

        this.cacheUnreadStats =
            config.getBoolean(
                "cache-unread-count",
                defaults.cacheUnreadStats());

        this.unreadStatCacheSize =
            config.getInt(
                "cache-unread-count-size",
                defaults.unreadStatCacheSize());
    }

    @Override
    public String pgQueue() {
        return pgQueue;
    }

    public T pgQueue(final String pgQueue) {
        this.pgQueue = pgQueue;
        return self();
    }

    @Override
    public String indexUri() {
        return indexUri;
    }

    public T indexUri(final String uri) {
        this.indexUri = uri;
        return self();
    }

    @Override
    public int usertypeBatchSize() {
        return usertypeBatchSize;
    }

    public T usertypeBatchSize(final int size) {
        this.usertypeBatchSize = size;
        return self();
    }

    @Override
    public ProducerClientConfig producerConfig() {
        return producerClientConfig;
    }

    public T producerConfig(final ProducerClientConfig config) {
        this.producerClientConfig = new ProducerClientConfigBuilder(config);
        return self();
    }

    @Override
    public Set<String> categories() {
        return categories;
    }

    public T categories(final Set<String> categories) {
        this.categories = new LinkedHashSet<>(categories);
        return self();
    }

    @Override
    public XivaConfig xivaConfig() {
        return xivaConfig;
    }

    public T xivaConfig(final XivaConfig config) {
        this.xivaConfig = new XivaConfigBuilder(config);
        return self();
    }

    @Override
    public HttpHostConfig proxyConfig() {
        return proxyConfig;
    }

    public T proxyConfig(final HttpHostConfig config) {
        this.proxyConfig = new HttpHostConfigBuilder(config);
        return self();
    }

    @Override
    public HttpTargetConfig searchClientConfig() {
        return searchClientConfig;
    }

    public T searchClientConfig(final HttpTargetConfig config) {
        this.searchClientConfig = new HttpTargetConfigBuilder(config);
        return self();
    }

    @Override
    public FilterSearchConfig filterSearchConfig() {
        return filterSearchConfig;
    }

    public T filterSearchConfig(final FilterSearchConfig config) {
        this.filterSearchConfig = new FilterSearchConfigBuilder(config);
        return self();
    }

    @Override
    public SearchMapConfig searchMapConfig() {
        return searchMapConfig;
    }

    public T searchMapConfig(final SearchMapConfig config) {
        this.searchMapConfig = new SearchMapConfigBuilder(config);
        return self();
    }

    @Override
    public int hoursSkipStaleRecords() {
        return hoursSkipStaleRecords;
    }

    public T hoursSkipStaleRecords(final int hours) {
        this.hoursSkipStaleRecords = hours;
        return self();
    }

    @Override
    public int unreadStatCacheSize() {
        return unreadStatCacheSize;
    }

    public T unreadStatCacheSize(
        final int unreadStatCacheSize)
    {
        this.unreadStatCacheSize = unreadStatCacheSize;
        return self();
    }

    @Override
    public boolean cacheUnreadStats() {
        return cacheUnreadStats;
    }

    public T cacheUnreadStats(
        final boolean cacheUnreadStats)
    {
        this.cacheUnreadStats = cacheUnreadStats;
        return self();
    }
}
