package ru.yandex.search.mail.xavier.store;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.logging.Level;

import org.apache.http.entity.ContentType;

import ru.yandex.dbfields.OracleFields;
import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;

import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.io.StringBuilderWriter;

import ru.yandex.json.dom.JsonList;
import ru.yandex.json.dom.JsonMap;
import ru.yandex.json.dom.JsonObject;

import ru.yandex.json.parser.JsonException;
import ru.yandex.json.writer.JsonValue;
import ru.yandex.json.writer.JsonWriter;

import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.mail.xavier.XavierContext;

import ru.yandex.search.mail.xavier.store.StoreXavierHandler.XivaCallback;

public class StoreUnreadCountCallback
    extends AbstractFilterFutureCallback<XavierData, Object>
{
    protected final XavierContext context;

    public StoreUnreadCountCallback(final XavierContext context) {
        super(context.callback());

        this.context = context;
    }

    private static StringBuilder buildEmail(
        final JsonMap from)
        throws JsonException
    {
        StringBuilder fromSb = new StringBuilder();
        fromSb.append("\\\"");
        fromSb.append(from.getString("displayName", ""));
        fromSb.append("\\\" ");
        fromSb.append(from.getString("local", ""));
        fromSb.append("@");
        fromSb.append(from.getString("domain", ""));
        return fromSb;
    }

    // CSOFF: MultipleStringLiterals
    // CSOFF: ParameterNumber
    private void message(
        final JsonWriter writer,
        final JsonMap envelope,
        final Map<String, Integer> counters,
        final Map<String, List<String>> categories)
        throws IOException, JsonException
    {
        writer.startObject();
        writer.key("hdr_status");
        writer.value("New");
        writer.key("lcn");
        writer.value(context.lcn());
        writer.key("mid");
        writer.value(envelope.getString(OracleFields.MID));
        writer.key("firstline");
        writer.value(envelope.getString(OracleFields.FIRSTLINE, ""));
        writer.key("sz");
        writer.value(envelope.getLong("size"));
        writer.key("hdr_subject");
        writer.value(envelope.getString(OracleFields.SUBJECT, ""));
        writer.key("fid");
        writer.value(envelope.getString(OracleFields.FID, ""));
        writer.key("operation");
        writer.value("insert");
        writer.key("thread_id");
        writer.value(envelope.getString("threadId", ""));
        writer.key("session_key");
        writer.value("");
        writer.key("method_id");
        writer.value("");
        writer.key("received_date");
        writer.value("");
        writer.key("uid");
        writer.value(context.user().prefix().toString());
        writer.key("lid");
        List<JsonObject> labels = envelope.asMap().getListOrNull("labels");
        if (labels != null) {
            StringBuilder sb = new StringBuilder();
            for (JsonObject label: labels) {
                if (sb.length() != 0) {
                    sb.append(",");
                }

                sb.append(label.asString());
            }

            writer.value(sb);
        } else {
            writer.value("");
        }

        writer.key("hdr_from");
        JsonList from = envelope.getListOrNull(OracleFields.FROM);
        if (from != null && from.size() > 0) {
            writer.value(buildEmail(from.get(0).asMap()));
        } else {
            writer.value("");
        }

        writer.key("hdr_to");
        JsonList to = envelope.getListOrNull(OracleFields.TO);
        if (to != null && to.size() > 0) {
            writer.value(buildEmail(to.get(0).asMap()));
        } else {
            writer.value("");
        }

        writer.key("fresh_count");
        writer.value(envelope.getString("freshCount", "0"));
        writer.key("hdr_message_id");
        writer.value(envelope.getString("messageId", ""));
        writer.key("fid_type");
        writer.value(envelope.getString("fidType", ""));

        writer.key("categories");
        writer.value(categories);

        writer.key("counters");
        writer.startObject();
        for (Map.Entry<String, Integer> entry : counters.entrySet()) {
            writer.key(entry.getKey());
            writer.startObject();
            writer.key("unread");
            writer.value(entry.getValue());
            writer.endObject();
        }

        writer.endObject();
        writer.endObject();
    }
    // CSON: ParameterNumber

    protected void process(
        final AsyncClient client,
        final XavierData data)
        throws IOException, JsonException, BadRequestException
    {
        context.session().logger().info("StoreCallback");
        JsonList envelopeList =
            data.envelopes().get("envelopes").asList();
        if (envelopeList.size() <= 0) {
            context.session().logger().info("No envelope found for request");
            context.callback().completed(null);
            return;
        }

        JsonMap envelope = envelopeList.get(0).asMap();

        context.session().logger().info(
            "Xiva request "
                + buildRequest(
                envelope.asMap(),
                data.counters(),
                data.categories())
                .toString());

        client.execute(
            context.xavier().config().xivaConfig().host(),
            buildRequest(envelope.asMap(), data.counters(), data.categories()),
            EmptyAsyncConsumerFactory.OK,
            context.session().listener().createContextGeneratorFor(
                client),
            new XivaCallback(context));
    }

    public BasicAsyncRequestProducerGenerator buildRequest(
        final JsonMap envelope,
        final Map<String, Integer> counters,
        final Map<String, List<String>> categories)
        throws IOException, JsonException, BadRequestException
    {
        QueryConstructor qc = new QueryConstructor("/v2/send?");
        qc.append("ttl", 0);
        qc.append(
            "token",
            context.xavier().config().xivaConfig().notifyToken());
        qc.append("user", context.prefix().toString());
        qc.append("event", "msearch_insert");
        qc.append("lcn", context.lcn().toString());

        StringBuilderWriter sbWriter = new StringBuilderWriter();
        try (JsonWriter writer = JsonWriter.create(sbWriter)) {
            writer.startObject();
            writer.key("payload");
            writer.startObject();
            writer.key("message");
            message(writer, envelope, counters, categories);
            writer.key("operation");
            writer.value("insert");
            writer.key("service");
            writer.value("msearch");
            writer.key("uid");
            writer.value(context.user().prefix().toString());
            writer.key("session_key");
            writer.value("");
            writer.key("lcn");
            writer.value(context.lcn());
            writer.key("tags");
            writer.startArray();
            writer.endArray();
            writer.key("version");
            writer.value("1");
            writer.key("raw_data");
            writer.startObject();
            writer.key("uid");
            writer.value(context.user().prefix().toString());
            writer.key("operation");
            writer.value("unsupported");
            writer.key("envelopes");
            writer.startArray();
            writer.value((JsonValue) envelope);
            writer.endArray();
            writer.endObject();
            writer.endObject();
            writer.endObject();
        }

        return new BasicAsyncRequestProducerGenerator(
            qc.toString(),
            sbWriter.toString(),
            ContentType.APPLICATION_JSON);
    }
    // CSON: MultipleStringLiterals

    @Override
    public void completed(
        final XavierData data)
    {
        context.session().logger().info(
            "Data prepared " + data.counters()
                + ' ' + data.categories().toString());
        // check empty
        boolean empty = true;
        for (List<String> categories: data.categories().values()) {
            if (!categories.isEmpty()) {
                empty = false;
                break;
            }
        }

        if (empty) {
            context.session().logger().info("No categories skipping");
            context.callback().completed(null);
            return;
        }

        AsyncClient client =
            context.xavier().xivaNotifyClient().adjust(
                context.session().context());
        try {
            process(client, data);
        } catch (IOException
            | JsonException
            | BadRequestException e)
        {
            context.session().logger().log(
                Level.WARNING,
                "Failed to construct xiva notification",
                e);
            context.callback().failed(e);
        }
    }
}
