package ru.yandex.search.mail.xavier.store;

import java.util.List;
import java.util.Map;
import java.util.TimerTask;
import java.util.logging.Level;

import org.apache.http.HttpException;

import org.apache.http.concurrent.FutureCallback;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.client.producer.QueueHostInfo;

import ru.yandex.http.proxy.AbstractProxySessionCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.json.parser.JsonException;

import ru.yandex.search.mail.xavier.FilterSearchStep;
import ru.yandex.search.mail.xavier.StaleChecker;

import ru.yandex.search.mail.xavier.UnreadCountCache.UserReadStat;

import ru.yandex.search.mail.xavier.Xavier;
import ru.yandex.search.mail.xavier.XavierContext;
import ru.yandex.search.mail.xavier.XavierHandler;
import ru.yandex.search.mail.xavier.XivaListConsumerFactory;

import ru.yandex.search.mail.xavier.proxy.CategoriesFetchStep;
import ru.yandex.search.mail.xavier.proxy.UnreadCountStep;

public class StoreXavierHandler implements XavierHandler {
    private static final long START_RETRY_DELAY = 200;
    private static final long MAX_RETRY_DELAY = 1700;
    private static final long INTERVAL = 300;

    private static final EmptyUnreadCallback EMPTY_UNREAD_CALLBACK =
        new EmptyUnreadCallback();

    private final Xavier xavier;
    private final UnreadCountStep unreadCountStep;
    private final CategoriesFetchStep categoriesFetchStep;
    private final FilterSearchStep filterSearchStep;

    public StoreXavierHandler(final Xavier xavier) {
        this.xavier = xavier;
        //this.proxySearchStep = new ProxySearchStep(xavier);
        this.unreadCountStep = new UnreadCountStep(xavier);
        this.categoriesFetchStep = new CategoriesFetchStep(xavier);
        this.filterSearchStep = new FilterSearchStep();
    }

    @Override
    public void handle(final XavierContext context)
        throws HttpException, JsonException
    {
        context.session().logger().info("Store handler");
        if (StaleChecker.isStaleNotify(context)) {
            context.callback().completed(null);
            return;
        }

        AsyncClient xivaClient =
            context.xavier().xivaListClient().adjust(
                context.session().context());
        BasicAsyncRequestProducerGenerator get =
            new BasicAsyncRequestProducerGenerator("/v2/list?service=mail&user="
                + context.prefix() + "&token="
                + context.xavier().config().xivaConfig().listToken());

        xivaClient.execute(
            context.xavier().config().xivaConfig().host(),
            get,
            XivaListConsumerFactory.OK,
            context.session().listener().createContextGeneratorFor(
                xivaClient),
            new XivaCheckCallback(context));
    }

    private void sendProducerRequest(
        final XavierContext context,
        final ProducerClient client,
        final long retryDelay)
        throws HttpException
    {
        client.executeWithInfo(
            context.user(),
            context.session().listener().createContextGeneratorFor(client),
            new PositionCallback(context, client, retryDelay));
    }

    private final class XivaCheckCallback
        extends AbstractProxySessionCallback<Boolean>
    {
        private final XavierContext context;

        XivaCheckCallback(final XavierContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final Boolean status) {
            if (!status) {
                context.session().logger().info("User is offline, ignoring");
                context.xavier().unreadCountCache().remove(
                    context.prefix().prefix());
                context.callback().completed(null);
            } else {
                ProducerClient client =
                    xavier.producerClient().adjust(context.session().context());

                try {
                    sendProducerRequest(context, client, START_RETRY_DELAY);
                } catch (Exception e) {
                    failed(e);
                }
            }
        }
    }

    private final class PositionCallback
        extends AbstractFilterFutureCallback<List<QueueHostInfo>, Object>
    {
        private final XavierContext context;
        private final ProducerClient producerClient;
        private final long retryDelay;

        private PositionCallback(
            final XavierContext context,
            final ProducerClient producerClient,
            final long retryDelay)
        {
            super(context.callback());

            this.context = context;
            this.producerClient = producerClient;
            this.retryDelay = retryDelay;
        }

        @Override
        public void completed(
            final List<QueueHostInfo> hostInfos)
        {
            boolean indexed =
                hostInfos.stream()
                    .filter(hi -> hi.queueId() >= context.queueId())
                    .count() > 0;

            context.session().logger().info("Hosts positions " + hostInfos);

            if (!indexed) {
//                if (retryDelay <= START_RETRY_DELAY
//                    && context.retryCount() == 0)
//                {
//                    UserReadStat stat =
//                        context.xavier().unreadCountCache().get(
//                            context.prefix().prefix());
//                    if (stat == null) {
//                        // disable feature
////                        unreadCountStep.fetch(
////                            context,
////                            EMPTY_UNREAD_CALLBACK,
////                            true);
//                    }
//                }
                if (retryDelay <= MAX_RETRY_DELAY) {
                    producerClient.scheduleRetry(
                        new ProducerRetryTimerTask(
                            context,
                            producerClient,
                            retryDelay),
                        retryDelay);
                } else {
                    context.callback().notReady();
                }
                return;
            }

            context.session().logger().info("Indexed !");
            FetchSession session;
            FutureCallback<XavierData> resultCalback =
                new StoreUnreadCountCallback(context);

            if (context.xavier().config().cacheUnreadStats()) {
                UserReadStat stat =
                    context.xavier().unreadCountCache().get(
                        context.prefix().prefix());
                if (stat != null) {
                    context.session().logger().info(
                        "From Cache " + context.prefix()
                            + ' ' + stat.asMap().toString());
                    session =
                        new CachedFetchSession(
                            context,
                            stat,
                            resultCalback);
                } else {
                    session = new CachingFetchSession(context, resultCalback);
                    unreadCountStep.fetch(
                        context,
                        new UnreadCountCallback(session));
                }
            } else {
                session = new FetchSession(resultCalback);
                unreadCountStep.fetch(
                    context,
                    new UnreadCountCallback(session));
            }

            filterSearchStep.fetch(context, new FilterSearchCallback(session));
            categoriesFetchStep.fetch(context, new CategoriesCallback(session));
        }
    }

    private final class ProducerRetryTimerTask extends TimerTask {
        private final XavierContext context;
        private final ProducerClient producerClient;
        private final long retryDelay;

        ProducerRetryTimerTask(
            final XavierContext context,
            final ProducerClient producerClient,
            final long retryDelay)
        {
            this.context = context;
            this.producerClient = producerClient;
            this.retryDelay = retryDelay;
        }

        @Override
        public void run() {
            context.session().logger().info("Retrying producer request after: "
                + retryDelay + " ms");
            try {
                sendProducerRequest(
                    context,
                    producerClient,
                    retryDelay + INTERVAL);
            } catch (HttpException e) {
                context.session().logger().log(Level.SEVERE, "Retry failed", e);
                context.callback().failed(e);
            }
        }
    }

    public static final class XivaCallback
        extends AbstractFilterFutureCallback<Void, Object>
    {
        private final XavierContext context;

        public XivaCallback(final XavierContext context) {
            super(context.callback());

            this.context = context;
        }

        @Override
        public void completed(final Void v) {
            context.session().logger().info("Xiva notify sent");
            context.callback().completed(v);
        }
    }

    private static final class EmptyUnreadCallback
        implements FutureCallback<Map<String, Integer>>
    {
        private EmptyUnreadCallback() {
        }

        @Override
        public void completed(
            final Map<String, Integer> map)
        {
        }

        @Override
        public void failed(final Exception e) {
        }

        @Override
        public void cancelled() {
        }
    }
}
