package ru.yandex.search.mail.xavier.update;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.TimerTask;
import java.util.logging.Level;

import org.apache.http.HttpException;

import org.apache.http.concurrent.FutureCallback;

import org.apache.http.entity.ContentType;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.client.producer.QueueHostInfo;

import ru.yandex.http.proxy.AbstractProxySessionCallback;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.EmptyAsyncConsumerFactory;
import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.io.StringBuilderWriter;

import ru.yandex.json.dom.JsonObject;

import ru.yandex.json.parser.JsonException;

import ru.yandex.json.writer.JsonWriter;

import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;

import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.mail.xavier.FilterSearchStep;
import ru.yandex.search.mail.xavier.StaleChecker;
import ru.yandex.search.mail.xavier.Xavier;
import ru.yandex.search.mail.xavier.XavierContext;
import ru.yandex.search.mail.xavier.XavierHandler;
import ru.yandex.search.mail.xavier.XivaListConsumerFactory;

import ru.yandex.search.mail.xavier.proxy.CategoriesFetchStep;
import ru.yandex.search.mail.xavier.proxy.UnreadCountStep;

import ru.yandex.search.mail.xavier.store.CachingFetchSession;
import ru.yandex.search.mail.xavier.store.CategoriesCallback;
import ru.yandex.search.mail.xavier.store.FetchSession;
import ru.yandex.search.mail.xavier.store.FilterSearchCallback;
import ru.yandex.search.mail.xavier.store.UnreadCountCallback;
import ru.yandex.search.mail.xavier.store.XavierData;

import ru.yandex.search.mail.xavier.usertype.UsertypeXavierHandler;

public class UpdateXavierHandler implements XavierHandler {
    private static final String CATEGORY_UPDATE_EVENT = "category_update";

    private static final int THOUSAND = 1000;
    private static final long START_RETRY_DELAY = 200;
    private static final long MAX_RETRY_DELAY = 1700;
    private static final long INTERVAL = 300;

    private final Xavier xavier;
    private final UnreadCountStep unreadCountStep;
    private final CategoriesFetchStep categoriesFetchStep;
    private final FilterSearchStep filterSearchStep;

    public UpdateXavierHandler(final Xavier xavier) {
        this.xavier = xavier;
        this.unreadCountStep = new UnreadCountStep(xavier);
        this.categoriesFetchStep = new CategoriesFetchStep(xavier);
        this.filterSearchStep = new FilterSearchStep();
    }

    @Override
    public void handle(final XavierContext context)
        throws HttpException, JsonException
    {
        if (StaleChecker.isStaleNotify(context)) {
            context.callback().completed(null);
            return;
        }

        AsyncClient xivaClient =
            context.xavier().xivaListClient().adjust(
                context.session().context());
        BasicAsyncRequestProducerGenerator get =
            new BasicAsyncRequestProducerGenerator("/v2/list?service=mail&user="
                + context.prefix() + "&token="
                + context.xavier().config().xivaConfig().listToken());

        xivaClient.execute(
            context.xavier().config().xivaConfig().host(),
            get,
            XivaListConsumerFactory.OK,
            context.session().listener().createContextGeneratorFor(
                xivaClient),
            new XivaCheckCallback(context));
    }

    private final class XivaCheckCallback
        extends AbstractProxySessionCallback<Boolean>
    {
        private final XavierContext context;

        XivaCheckCallback(final XavierContext context) {
            super(context.session());
            this.context = context;
        }

        @Override
        public void completed(final Boolean online) {
            if (!online) {
                context.session().logger().info("User is offline, ignoring");
                context.xavier().unreadCountCache().remove(
                    context.prefix().prefix());
                context.callback().completed(null);
            } else {
                ProducerClient client =
                    xavier.producerClient().adjust(context.session().context());

                try {
                    sendProducerRequest(context, client, START_RETRY_DELAY);
                } catch (Exception e) {
                    failed(e);
                }
            }
        }
    }

    private void sendProducerRequest(
        final XavierContext context,
        final ProducerClient client,
        final long retryDelay)
        throws HttpException
    {
        client.executeWithInfo(
            context.user(),
            context.session().listener().createContextGeneratorFor(client),
            new PositionCallback(context, client, retryDelay));
    }

    private final class ProducerRetryTimerTask extends TimerTask {
        private final XavierContext context;
        private final ProducerClient producerClient;
        private final long retryDelay;

        ProducerRetryTimerTask(
            final XavierContext context,
            final ProducerClient producerClient,
            final long retryDelay)
        {
            this.context = context;
            this.producerClient = producerClient;
            this.retryDelay = retryDelay;
        }

        @Override
        public void run() {
            context.session().logger().info(
                "Retrying producer request after: "
                    + retryDelay + " ms");
            try {
                sendProducerRequest(
                    context,
                    producerClient,
                    retryDelay + INTERVAL);
            } catch (HttpException e) {
                context.session().logger().log(Level.SEVERE, "Retry failed", e);
                context.callback().failed(e);
            }
        }
    }

    private final class PositionCallback
        extends AbstractFilterFutureCallback<List<QueueHostInfo>, Object>
    {
        private final XavierContext context;
        private final ProducerClient producerClient;
        private final long retryDelay;

        private PositionCallback(
            final XavierContext context,
            final ProducerClient producerClient,
            final long retryDelay)
        {
            super(context.callback());

            this.context = context;
            this.producerClient = producerClient;
            this.retryDelay = retryDelay;
        }

        @Override
        public void completed(final List<QueueHostInfo> hostInfos) {
            boolean indexed =
                hostInfos.stream()
                    .filter(hi -> hi.queueId() >= context.queueId())
                    .count() > 0;

            context.session().logger().info("Hosts positions " + hostInfos);

            if (!indexed) {
                if (retryDelay <= MAX_RETRY_DELAY) {
                    producerClient.scheduleRetry(
                        new ProducerRetryTimerTask(
                            context,
                            producerClient,
                            retryDelay),
                        retryDelay);
                } else {
                    context.callback().notReady();
                }
                return;
            }

            String targetField = null;

            try {
                targetField = ValueUtils.asStringOrNull(
                    context.json().get(UsertypeXavierHandler.TARGET_FIELD));
            } catch (JsonUnexpectedTokenException jute) {
                context.session().logger().log(
                    Level.WARNING,
                    "Broken payload",
                    jute);
            }

            FutureCallback<XavierData> cb;
            if (targetField == null) {
                try {
                    if (StaleChecker.isStaleNotify(context)) {
                        context.callback().completed(null);
                        return;
                    }
                } catch (BadRequestException bre) {
                    context.session().logger().log(
                        Level.WARNING,
                        "Failed stale check",
                        bre);
                }

                cb = new StatusChangeCallback(context);
            } else {
                context.session().logger().info(
                    "Target field is not empty " + targetField);
                cb = new UpdatePreparedCallback(context, targetField);
            }

            FetchSession session;
            if (context.xavier().config().cacheUnreadStats()) {
                session = new CachingFetchSession(context, cb);
            } else {
                session = new FetchSession(cb);
            }

            unreadCountStep.fetch(
                context,
                new UnreadCountCallback(session));
            filterSearchStep.fetch(context, new FilterSearchCallback(session));
            categoriesFetchStep.fetch(context, new CategoriesCallback(session));
        }
    }

    // CSOFF: MultipleStringLiterals
    private static final class UpdatePreparedCallback
        extends AbstractFilterFutureCallback<XavierData, Object>
    {
        private final XavierContext context;

        private UpdatePreparedCallback(
            final XavierContext context,
            final String target)
        {
            super(context.callback());

            this.context = context;
        }

        // CSOFF: MultipleStringLiterals
        private void message(
            final JsonWriter writer,
            final Map<String, Integer> counters)
            throws IOException, JsonException
        {
            writer.startObject();
            writer.key("mids");
            writer.value(context.mids());

            writer.key("operation");
            writer.value("category_mails");

            writer.key("counters");
            writer.startObject();
            for (Map.Entry<String, Integer> entry : counters.entrySet()) {
                writer.key(entry.getKey());
                writer.startObject();
                writer.key("unread");
                writer.value(entry.getValue());
                writer.endObject();
            }
            writer.endObject();

            writer.endObject();
        }

        private BasicAsyncRequestProducerGenerator buildRequest(
            final JsonObject envelopes,
            final Map<String, Integer> counters,
            final Map<String, List<String>> categories)
            throws IOException, JsonException, BadRequestException
        {
            QueryConstructor qc = new QueryConstructor("/v2/send?");
            qc.append("ttl", 0);
            qc.append(
                "token",
                context.xavier().config().xivaConfig().notifyToken());
            qc.append("user", context.prefix().toString());
            qc.append("event", CATEGORY_UPDATE_EVENT);

            StringBuilderWriter sbWriter = new StringBuilderWriter();
            try (JsonWriter writer = JsonWriter.create(sbWriter)) {
                writer.startObject();
                writer.key("payload");
                writer.startObject();
                writer.key("message");
                message(writer, counters);
                writer.key("operation");
                writer.value("category mails");
                writer.key("service");
                writer.value("msearch");
                writer.key("uid");
                writer.value(context.user().prefix().toString());
                writer.key("tags");
                writer.startArray();
                writer.endArray();
                writer.key("version");
                writer.value("1");

                writer.key("categories");
                writer.value(categories);

                writer.key("raw_data");
                message(writer, counters);
                //writer.key("uid");
                //writer.value(context.user().prefix().toString());
                //writer.key("envelopes");
                //writer.value(envelopes);
                writer.endObject();
                writer.endObject();
            }

            return new BasicAsyncRequestProducerGenerator(
                qc.toString(),
                sbWriter.toString(),
                ContentType.APPLICATION_JSON);
        }
        // CSON: MultipleStringLiterals

        @Override
        public void completed(final XavierData data) {
            context.session().logger().info("Data gathered, build xiva notify");
            AsyncClient client =
                context.xavier().xivaNotifyClient().adjust(
                    context.session().context());

            try {
                double userTypeChangeTime =
                    context.session().params().getDouble(
                        "usertype_change_date", -1.0);

                client.execute(
                    context.xavier().config().xivaConfig().host(),
                    buildRequest(
                        data.envelopes(),
                        data.counters(),
                        data.categories()),
                    EmptyAsyncConsumerFactory.OK,
                    context.session().listener().createContextGeneratorFor(
                        client),
                    new XivaCallback(context, userTypeChangeTime));
            } catch (Exception e) {
                context.session().logger().log(
                    Level.WARNING,
                    "Failed to construct xiva notification",
                    e);
                context.callback().failed(e);
            }
        }
    }

    private static final class XivaCallback
        extends AbstractFilterFutureCallback<Void, Object>
    {
        private final XavierContext context;
        private final double userTypeChangeTime;

        private XivaCallback(
            final XavierContext context,
            final double userTypeChangeTime)
        {
            super(context.callback());

            this.context = context;
            this.userTypeChangeTime = userTypeChangeTime;
        }

        @Override
        public void completed(final Void v) {
            context.session().logger().info("Xiva notify sent");
            if (userTypeChangeTime > 0) {
                long lag =
                    (long) (System.currentTimeMillis()
                        - this.userTypeChangeTime * THOUSAND);

                context.xavier().staters().userTypeChange().accept(lag);
            }

            context.callback().completed(v);
        }
    }
}
