package ru.yandex.search.mail.xavier.usertype;

import java.io.IOException;
import java.util.logging.Level;

import org.apache.http.entity.ContentType;

import ru.yandex.client.producer.ProducerClient;

import ru.yandex.dbfields.ChangeType;
import ru.yandex.dbfields.OracleFields;
import ru.yandex.dbfields.PgFields;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.BasicAsyncResponseConsumerFactory;

import ru.yandex.io.StringBuilderWriter;

import ru.yandex.json.writer.JsonWriter;

import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.mail.xavier.Xavier;
import ru.yandex.search.mail.xavier.XavierContext;

import ru.yandex.search.result.SearchDocument;
import ru.yandex.search.result.SearchResult;

public class UserTypeSearchCallback
    extends AbstractFilterFutureCallback<SearchResult, Object>
{
    private static final String MDB = "mdb";

    private static final double THOUSAND = 1000.0;

    private final XavierContext context;
    private final Xavier xavier;
    private final String target;

    public UserTypeSearchCallback(
        final XavierContext context,
        final String target)
    {
        super(context.callback());

        this.context = context;
        this.xavier = context.xavier();
        this.target = target;
    }

    @Override
    public void completed(final SearchResult result) {
        double operationDate = System.currentTimeMillis() / THOUSAND;

        QueryConstructor uri = new QueryConstructor(xavier.config().indexUri());

        context.session().logger().info(
            "Found "
                + result.hitsCount()
                + " documents for reindexation");

        try {
            uri.append("service", xavier.config().pgQueue());
            uri.append("prefix", context.prefix().toString());
            uri.append(MDB, context.session().params().getString(MDB));
            uri.append("uid", context.prefix().toString());
            uri.append("usertype_target", target);
            uri.append("xavier", "true");
        } catch (BadRequestException bre) {
            context.session().logger().log(
                Level.SEVERE,
                "Unable construct producer request ",
                bre);
            failed(bre);
            return;
        }

        StringBuilderWriter sbWriter = new StringBuilderWriter();
        try (JsonWriter writer = JsonWriter.create(sbWriter)) {
            writer.startObject();
            writer.key(PgFields.UID);
            writer.value(context.prefix());
            writer.key(OracleFields.LCN);
            writer.value(context.queueId());
            writer.key(PgFields.CHANGE_TYPE);
            writer.value(ChangeType.UPDATE);
            writer.key("operation_date");
            writer.value(operationDate);
            writer.key(UsertypeXavierHandler.TARGET_FIELD);
            writer.value(target);
            if (context.operationDate() != null) {
                writer.key("usertype_change_date");
                writer.value(context.operationDate());
            }

            writer.key(PgFields.CHANGED);
            writer.startArray();
            for (SearchDocument doc: result.hitsArray()) {
                writer.startObject();
                writer.key(PgFields.MID);
                writer.value(doc.attrs().get(PgFields.MID));
                writer.endObject();
            }

            writer.endArray();
            writer.endObject();
        } catch (IOException ioe) {
            context.session().logger().log(
                Level.SEVERE,
                "Can not create json " + result.toString(),
                ioe);
            context.callback().failed(ioe);
            return;
        }

        context.session().logger().info(
            "Reindexing producer request " + uri.toString()
                + " data " + sbWriter.toString());

        ProducerClient client =
            xavier.producerClient().adjust(context.session().context());

        boolean hasExtra = result.hitsCount() > result.hitsArray().size();
        if (!hasExtra) {
            context.session().logger().info("Usertype change last request");
        } else {
            context.session().logger().info(
                "Still "
                    + (result.hitsCount() - result.hitsArray().size())
                    + " records to process");
        }

        context.session().logger().info(
            "Usertype " + uri.toString()
                + " request \n " + sbWriter.toString());
        client.execute(
            xavier.config().producerConfig().host(),
            new BasicAsyncRequestProducerGenerator(
                uri.toString(),
                sbWriter.toString(),
                ContentType.APPLICATION_JSON),
            BasicAsyncResponseConsumerFactory.OK,
            new UserTypeProducerCallback(context, client, hasExtra));
    }
}
