package ru.yandex.search.mail.xavier.usertype;

import java.util.ArrayList;
import java.util.List;
import java.util.TimerTask;
import java.util.logging.Level;
import java.util.stream.Collectors;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;

import ru.yandex.client.producer.ProducerClient;
import ru.yandex.client.producer.QueueHostInfo;

import ru.yandex.http.util.AbstractFilterFutureCallback;
import ru.yandex.http.util.BadRequestException;

import ru.yandex.http.util.nio.BasicAsyncRequestProducerGenerator;
import ru.yandex.http.util.nio.client.AsyncClient;

import ru.yandex.json.parser.JsonException;

import ru.yandex.json.xpath.JsonUnexpectedTokenException;
import ru.yandex.json.xpath.ValueUtils;

import ru.yandex.parser.uri.QueryConstructor;

import ru.yandex.search.mail.xavier.Xavier;
import ru.yandex.search.mail.xavier.XavierContext;
import ru.yandex.search.mail.xavier.XavierHandler;

import ru.yandex.search.proxy.SearchResultConsumerFactory;
import ru.yandex.search.request.util.SearchRequestText;

public class UsertypeXavierHandler implements XavierHandler {
    public static final String FROM_FIELD = "hdr_from";
    public static final String TARGET_FIELD = "target_category";

    private static final long START_RETRY_DELAY = 200;
    private static final long MAX_RETRY_DELAY = 1700;
    private static final long INTERVAL = 300;
    private static final String PREFIX = "prefix";

    private final Xavier xavier;

    public UsertypeXavierHandler(final Xavier xavier) {
        this.xavier = xavier;
    }

    @Override
    public void handle(final XavierContext context)
        throws HttpException, JsonException
    {
        ProducerClient client =
            xavier.producerClient().adjust(context.session().context());

        client.executeWithInfo(
            context.user(),
            context.session().listener().createContextGeneratorFor(client),
            new PositionCallback(context, client, START_RETRY_DELAY));
    }

    private void sendProducerRequest(
        final XavierContext context,
        final ProducerClient client,
        final long retryDelay)
        throws HttpException
    {
        client.executeWithInfo(
            context.user(),
            context.session().listener().createContextGeneratorFor(client),
            new PositionCallback(context, client, retryDelay));
    }

    private final class PositionCallback
        extends AbstractFilterFutureCallback<List<QueueHostInfo>, Object>
    {
        private final XavierContext context;
        private final ProducerClient producerClient;
        private final long retryDelay;

        private PositionCallback(
            final XavierContext context,
            final ProducerClient producerClient,
            final long retryDelay)
        {
            super(context.callback());

            this.context = context;
            this.producerClient = producerClient;
            this.retryDelay = retryDelay;
        }

        @Override
        public void completed(final List<QueueHostInfo> hostInfos) {
            List<HttpHost> hosts =
                hostInfos.stream()
                    .filter(hi -> hi.queueId() >= context.queueId())
                    .map(QueueHostInfo::host)
                    .collect(
                        Collectors.toCollection(
                            () -> new ArrayList<>(hostInfos.size())));

            if (hosts.isEmpty()) {
                if (retryDelay <= MAX_RETRY_DELAY) {
                    producerClient.scheduleRetry(
                        new ProducerRetryTimerTask(
                            context,
                            producerClient,
                            retryDelay),
                        retryDelay);
                } else {
                    context.callback().notReady();
                }

                return;
            }

            String target;
            QueryConstructor request = new QueryConstructor("/search?");

            try {
                target =
                    ValueUtils.asStringOrNull(context.json().get(TARGET_FIELD));

                if (target == null) {
                    target =
                        ValueUtils.asString(context.json().get("userType"));
                }

                String from =
                    ValueUtils.asString(context.json().get(FROM_FIELD));

                StringBuilder text = new StringBuilder("hdr_from_normalized:");
                text.append(SearchRequestText.fullEscape(from, false));
                request.append(PREFIX, String.valueOf(context.prefix()));
                request.append("text", text.toString());
                request.append("group", "mid");
                request.append("merge_func", "none");
                request.append("get", "mid,lcn");
                request.append(
                    "length",
                    String.valueOf(
                        context.xavier().config().usertypeBatchSize()));
                request.append("sort", "received_date");
            } catch (BadRequestException | JsonUnexpectedTokenException e) {
                context.callback().failed(e);
                return;
            }

            AsyncClient client = xavier
                .searchClient()
                .adjust(context.session().context());

            context.session().logger().info(
                "Fetching for reindexation " + request.toString());

            client.execute(
                hosts,
                new BasicAsyncRequestProducerGenerator(request.toString()),
                SearchResultConsumerFactory.OK,
                context.session().listener()
                    .createContextGeneratorFor(client),
                new UserTypeSearchCallback(context, target));
        }
    }

    private final class ProducerRetryTimerTask extends TimerTask {
        private final XavierContext context;
        private final ProducerClient producerClient;
        private final long retryDelay;

        ProducerRetryTimerTask(
            final XavierContext context,
            final ProducerClient producerClient,
            final long retryDelay)
        {
            this.context = context;
            this.producerClient = producerClient;
            this.retryDelay = retryDelay;
        }

        @Override
        public void run() {
            context.session().logger().info(
                "Retrying producer request after: " + retryDelay + " ms");
            try {
                sendProducerRequest(
                    context,
                    producerClient,
                    retryDelay + INTERVAL);
            } catch (HttpException e) {
                context.session().logger().log(Level.SEVERE, "Retry failed", e);
                context.callback().failed(e);
            }
        }
    }
}
