package ru.yandex.zora.proxy;

import java.io.IOException;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.List;

import org.apache.http.HttpException;
import org.apache.http.HttpHost;
import org.apache.http.HttpRequest;

import org.apache.http.HttpStatus;
import org.apache.http.concurrent.FutureCallback;
import org.apache.http.nio.protocol.BasicAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncExchange;
import org.apache.http.nio.protocol.HttpAsyncRequestConsumer;
import org.apache.http.nio.protocol.HttpAsyncRequestHandler;

import org.apache.http.protocol.HttpContext;

import ru.yandex.http.proxy.BasicProxySession;
import ru.yandex.http.proxy.ProxySession;

import ru.yandex.http.util.BadRequestException;
import ru.yandex.http.util.MultiFutureCallback;

public class ImageHandler implements HttpAsyncRequestHandler<HttpRequest> {
    private static final int MIN_EXT_SIZE = 3;
    private static final int MAX_EXT_SIZE = 4;

    private final ZoraProxy server;
    private final ZoraCache cache;

    public ImageHandler(final ZoraProxy server) {
        this.server = server;
        this.cache = server.cache();
    }

    @Override
    public HttpAsyncRequestConsumer<HttpRequest> processRequest(
        final HttpRequest httpRequest,
        final HttpContext httpContext)
        throws HttpException, IOException
    {
        return new BasicAsyncRequestConsumer();
    }

    private int handle(
        final ProxySession session,
        final ImageResultPrinter printer,
        final List<String> urls)
        throws BadRequestException
    {
        MultiFutureCallback<Image> mfcb = new MultiFutureCallback<>(printer);

        int scheduled = 0;
        for (String urlStr: urls) {
            FutureCallback<Image> callback = mfcb.newCallback();
            Image image = cache.downloaded().getIfPresent(urlStr);
            if (image != null) {
                server.cacheHit(true);
                session.logger().info(urlStr + " Found in cache");
                callback.completed(image);
                continue;
            }

            server.cacheHit(false);

            DownloadTask task = new DownloadTask(urlStr);
            DownloadTask current = cache.pending().putIfAbsent(urlStr, task);
            if (current != null) {
                synchronized (current) {
                    if (!current.done()) {
                        current.subscribeForCancellation(callback);
                    } else {
                        if (current.result() == null) {
                            callback.failed(current.error());
                        } else {
                            callback.completed(current.result());
                        }
                    }
                }
            } else {
                ImageWaitingLoadCallback tlCallback =
                    new ImageWaitingLoadCallback(
                        callback,
                        session,
                        cache,
                        task);

                URL url;
                HttpHost host;
                try {
                    url = new URL(urlStr);
                    host = new HttpHost(url.getHost());
                } catch (MalformedURLException | IllegalArgumentException e) {
                    session.logger().warning("Malformed image url " + urlStr);
                    tlCallback.failed(e);
                    continue;
                }

                if (!checkUrl(url)) {
                    session.logger().info("Suspicious image " + urlStr);
                    server.suspiciousImagesDownloader().add(
                        host,
                        session,
                        new MissfitsWaitingLoadCallback(tlCallback));
                    scheduled += 1;

                    continue;
                }

                try {
                    server.scheduler().schedule(session, urlStr, tlCallback);
                    scheduled += 1;
                } catch (MalformedURLException mue) {
                    session.logger().warning("Malformed url " + urlStr);
                    tlCallback.failed(mue);
                }
            }
        }

        mfcb.done();

        return scheduled;
    }

    @Override
    public void handle(
        final HttpRequest request,
        final HttpAsyncExchange exchange,
        final HttpContext context)
        throws HttpException, IOException
    {
        ProxySession session = new BasicProxySession(server, exchange, context);
        ImageResultPrinter printer = new ImageResultPrinter(session);

        List<String> urls = session.params().getAll("url");
        //String urlStr = session.params().getString("url");
        if (urls.isEmpty()) {
            throw new BadRequestException("No url parameter supplied");
        }

        int scheduled = handle(session, printer, urls);

        boolean nowait = session.params().getBoolean("nowait", false);
        if (nowait) {
            if (!exchange.isCompleted()) {
                if (scheduled > 0) {
                    printer.completed(HttpStatus.SC_ACCEPTED);
                } else {
                    printer.completed(HttpStatus.SC_LOCKED);
                }
            }
        }
    }

    private boolean checkUrl(final URL url) {
        if (url.getPath() != null
            && (url.getQuery() == null || url.getQuery().isEmpty()))
        {
            String[] split = url.getPath().split("\\.");
            if (split.length > 1) {
                String last = split[split.length - 1];
                if (last.length() >= MIN_EXT_SIZE
                    && last.length() <= MAX_EXT_SIZE
                    && !"gif".equalsIgnoreCase(last))
                {
                    return true;
                }
            }
        }

        return false;
    }
}
